/*
  name: mywidget.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


// inclusion principale
#include "mywidget.h" 
// inclusions canvas
#include "objectcanvas.h" 
#include "transitioncanvas.h"
// inclusions boites de dialogue
#include "sheetdialog.h"
#include "statedialog.h"
#include "transitiondialog.h"
#include "blockdialog.h"
#include "initvaluedialog.h"
#include "selectmemberdialog.h"
#include "orderdialog.h"
#include "execdialog.h"
// inclusions lecture/ecriture
#include "projectwriter.h"
#include <metalibrary/metaconfigwriter.h>
#include <metalibrary/metaconfigreader.h>

// inclusion headers QT
#include <qapplication.h>
#include <qinputdialog.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qstatusbar.h>
#include <qfiledialog.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qpicture.h>
#include <qmessagebox.h> 

using namespace std;


MyWidget::MyWidget( QWidget *parent, const char *name )
    : QMainWindow( parent, name )
{


     Sheet = 0;
     State = new StateMachineView( this , "State Machine" );
		     
    /* Cration de notre menu popup */
    QPopupMenu *file = new QPopupMenu( this );
    Q_CHECK_PTR( file );
    //new project
    file->insertItem( "&New Project", this, SLOT(clear()), CTRL+Key_N );
    file->insertSeparator();
    file->insertItem( "&Open Project", this, SLOT(Import()), CTRL+Key_O );
    file->insertItem( "&Save Project", this, SLOT(ExportDial()), CTRL+Key_S );
    file->insertSeparator();
    file->insertItem( "Save Canvas", this, SLOT(saveCanvas()) );
    file->insertSeparator();
    file->insertItem( "Global defines", this, SLOT(globalDefines()));
    file->insertSeparator();
    file->insertItem( "&Run project", this, SLOT(run()), CTRL+Key_R );
    //file->insertItem( "&Close", this, SLOT(closeDoc()), CTRL+Key_W );
    file->insertSeparator();
    file->insertItem( "E&xit",  qApp, SLOT(quit()), CTRL+Key_Q );
    
    menuSheet = new QPopupMenu( this );
    Q_CHECK_PTR( menuSheet );
    menuSheet->insertItem( "add sheet",  this, SLOT(addSheet()),0, AddSheet);
    menuSheet->insertItem( "del sheet",  this, SLOT(delSheet()),0, DelSheet);
    menuSheet->insertItem( "rename sheet" , this, SLOT(renameSheet()),0, RenSheet);
    menuSheet->insertSeparator();
    menuSheet->insertItem( "add &Library",  this, SLOT(addLibrary()), CTRL+Key_L, AddLib );
    menuSheet->insertItem( "add &Object",  this, SLOT(addObjectDial()), CTRL+Key_B, AddObj );
    menuSheet->insertSeparator();
    menuSheet->insertItem( "add &Init",  this, SLOT(addInitDial()), CTRL+Key_I, AddIni );
    menuSheet->insertItem( "add &Wire",  this, SLOT(addWireDial()), CTRL+Key_W, AddWir );
    menuSheet->insertItem( "add meta-signal", this, SLOT(addMetaSignal()), CTRL+Key_O, AddMsi);
    menuSheet->insertItem( "add meta-slot", this, SLOT(addMetaSlot()), CTRL+Key_I, AddMsl);
			  
    menuSheet->insertSeparator();
    menuSheet->insertItem( "Order Pre-connections", this, SLOT(orderPreInitDial()), CTRL+Key_1, OrderPre );
    menuSheet->insertItem( "Order Post-connections", this, SLOT(orderPostInitDial()), CTRL+Key_2, OrderPost );
    menuSheet->insertItem( "Order connections", this, SLOT(orderWireDial()), CTRL+Key_3, OrderWire );


    menuState = new QPopupMenu( this );
    Q_CHECK_PTR( menuState );
    menuState->insertItem( "add &State",  this, SLOT(addStateDial()),0, AddSta );
    menuState->insertItem( "add &Transition",  this, SLOT(addTransitionDial()),0, AddTra );


    menuZoom = new QPopupMenu( this);
    menuZoom->insertItem("Refresh",this,SLOT(refresh()), CTRL+Key_R);
    menuZoom->insertSeparator();    
    menuZoom->insertItem("Zoom in",this,SLOT(zoom(int)), Key_Plus, ZoomIn);
    menuZoom->insertItem("Zoom out",this,SLOT(zoom(int)), Key_Minus, ZoomOut);
    menuZoom->insertItem("Zoom 1:1",this,SLOT(zoom(int)), Key_1, Zoom11);
    
    view = new QPopupMenu( this );
    Q_CHECK_PTR( view );

    view->setCheckable ( true );
    view->insertItem( "StateMachine" , this, SLOT(StateMachine()),0, StateView);
    view->insertSeparator();
    QLabel *caption = new QLabel( "<font color=blue size=3><u><b>"
				  "sheet(s)</b></u></font>", view );
    caption->setAlignment( Qt::AlignCenter );
    numIndex = LastSheet;
    view->insertItem( caption , numIndex );
    numIndex += 1;
    connect( view, SIGNAL(activated(int)), this, SLOT(selectSheet(int))); 
    indexSheet = numIndex;

    
    /* Cration de notre menu dans lequel viendrons s'ajouter les menus popup */
    Menu = new QMenuBar( this );
    Q_CHECK_PTR( Menu );
    Menu->insertItem( "&File", file );
    Menu->insertItem( "&Edit", view );
    Menu->insertItem( "&View", menuZoom );
    Menu->insertItem( "&State Machine", menuState );
    Menu->insertItem( "&Sheet", menuSheet );
    Menu->insertItem( "About", this, SLOT(about()));

    viewActive = new QLabel( Menu );
    //Parametre de la longueur minimal du label qui affiche la view en cours
    viewActive->setMinimumWidth(250);
    viewActive->setMaximumHeight(20);
    viewActive->setAlignment(Qt::AlignJustify);

    /* Configarution spaciale de notre fentre */
    //setCentralWidget(State);
    QStatusBar* bar = new QStatusBar(this);
    bar->addWidget(viewActive);

    /*connect( Sheet, SIGNAL(getLibrary(MetaLibrary**)), this, SLOT(giveLibrary(MetaLibrary**)));
    */

    contextMenu = new QPopupMenu( this );
    Q_CHECK_PTR( contextMenu );
    
    connect( State, SIGNAL(changeSheet(QString)), this, SLOT(changeSelect(QString)));
    
    setMode(1);
    adjustSize();
    dSheet.setAutoDelete(true);
    defdiag = new DefinesDialog(this, "Variables definition");
}


MyWidget::~MyWidget()
{
     delete contextMenu;
}


void MyWidget::about()
{
     QMessageBox::information(this, "About", "MetaEditor : a graphical editor for ARCS\n$Rev: 199 $");
}



void MyWidget::resetPointers()
{
     ObjectSelect = NULL;
     //StateCanvas *StateTerminal;
     InitSelect = NULL;
     WireSelect = NULL;
}



void MyWidget::contextMenuEvent( QContextMenuEvent * e)
{
     /* Menu s'affichant que lors d'un click droit de la souris */
     // Cration du menu popup
     //QLabel *caption = new QLabel( "<font color=red size=5><u>"
//				   "Context Menu</u></font>", this );
//     caption->setAlignment( Qt::AlignCenter );
//     contextMenu->insertItem( caption );
     contextMenu->clear();
     //int index;
     Canvas->update();

     //StateTerminal = NULL;
     InitSelect = NULL;
     WireSelect = NULL;

     if ( State->isVisible() ) {

	  QCanvasItemList list = Canvas->collisions((State->inverseWorldMatrix()).map(State->viewportToContents(e->pos())));
	  if (! list.empty())
	  {
	       for( uint i = 0; i < list.count(); i++)
	       {
		    if (list[i]->rtti() == RTTI_STATECANVAS )
		    {
			 StateSelect = ((StateCanvas*)list[i]);
			 idStateSelect = StateSelect->getName();
			 contextMenu->insertItem( "Set as Initial State", this, SLOT(setInitialState()));
			 contextMenu->insertItem( "Set as Terminal State", this, SLOT(setTerminalState()));
			 contextMenu->insertSeparator();
			 contextMenu->insertItem( "Del State", this, SLOT(delState()));
			 break;		    
		    }
		    
		    if (list[i]->rtti() == RTTI_TRANSITIONCANVAS )
		    {
			 TransitionSelect = ((TransitionCanvas*)list[i]);
			 contextMenu->insertItem( "del Transition", this, SLOT(delTransition()));
			 break;
		    }	    
	       }
	  }
	  else {
	       contextMenu->insertItem( "add &State",  this, SLOT(addStateDial()),0, AddSta );
	       contextMenu->insertItem( "add &Transition",  this, SLOT(addTransitionDial()),0, AddTra );
	  }
     }
     else {

	  //idObjectToken = "";
          QPoint p = e->pos();
          if (Sheet)
	       p = (Sheet->inverseWorldMatrix()).map(Sheet->viewportToContents(e->pos()));

	  QCanvasItemList list = Canvas->collisions(p);
	  if (! list.empty())
	  {
	       for( uint i = 0; i < list.count(); i++)
	       {
		    if (list[i]->rtti() == RTTI_OBJECTCANVAS )
		    {
			 ObjectSelect = ((ObjectCanvas*)list[i]);
			 idObjectToken = ObjectSelect->getObjectID();
			 contextMenu->insertItem( "Set as TokenSender", this, SLOT(setTokenSender()));
			 contextMenu->insertSeparator();
			 contextMenu->insertItem( "Delete Object", this, SLOT(delObject())); 
			 contextMenu->insertItem( "Switch persistence", this, SLOT(switchObjectPersistence()));

			 break;		    
		    }
		    if (list[i]->rtti() == RTTI_INITCANVAS )
		    {
			 InitSelect = ((InitCanvas*)list[i]);
			 contextMenu->insertItem( "Del Init", this, SLOT(delInit()));
			 contextMenu->insertItem( "Modify Init", this, SLOT(modifyInit()));
			 break;		    
		    }
		    if (list[i]->rtti() == RTTI_LINEWIRE )
		    {
			 WireSelect = (WireCanvas*)( ((LineWire*)list[i])->getOwner() );
			 contextMenu->insertItem( "Del Wire", this, SLOT(delWire()));
			 break;		    
		    }
		    if (list[i]->rtti() == RTTI_IOCANVAS )
		    {
			 IOSelect = (IOCanvas*)list[i];
			 contextMenu->insertItem("Del IO", this, SLOT(delIO()));
			 break;
		    }
	       }
	  }
	  else {
	       contextMenu->insertItem( "Add &Library", this, SLOT(addLibrary()),    CTRL+Key_L );
	       contextMenu->insertItem(  "Add &Object", this, SLOT(addObjectDial()), CTRL+Key_O );
	       contextMenu->insertItem(    "Add &Wire", this, SLOT(addWireDial()),   CTRL+Key_W );
	       contextMenu->insertItem(    "Add &Init", this, SLOT(addInitDial()),   CTRL+Key_I );
	       contextMenu->insertSeparator();
	       contextMenu->insertItem(    "Order Pre-connections", this, SLOT(orderPreInitDial()) );
	       contextMenu->insertItem(    "Order Post-connections", this, SLOT(orderPostInitDial()) );
	       contextMenu->insertItem(    "Order connections", this, SLOT(orderWireDial()) );
	  	  
	  }
     }
     //cout<<"ObjectCanvas"<<endl;
     // Ajouter de liens dans notre menu popup
     
     contextMenu->exec( QCursor::pos() );
     
     // Destruction du menu popup
     // delete contextMenu;
}


void MyWidget::switchObjectPersistence()
{
     if (ObjectSelect)
	  ObjectSelect->setObjectPersistence(true);
     update();
}

void MyWidget::setInitialState()
{
     State->setInitialState(idStateSelect);
     Canvas->update();
}

void MyWidget::setTerminalState()
{
     State->setTerminalState(idStateSelect);
     Canvas->update();
}

void MyWidget::setTokenSender()
{
     QStrList sgn = ml.getObjectSignals(ObjectSelect->getMetaObject()->getClassName(),true);
     for (uint i=0; i < sgn.count(); i++ )
     {
	  if ( sgn.at(i) == QString("sendToken(QString)") ) {
	       Sheet->setTokenSender(idObjectToken);
	       Canvas->update();
	       return;
	  }	       
     }
     QMessageBox::information( this, "Token Sender Definitition","This object doesn't possess signal sendToken(QString)");
}

void MyWidget::delObject()
{
     // on parcours chaque wire et on les del
     // remove des wire des signaux
     if (Sheet)
	  Sheet->resetPointers();

     
     //WireSelect = ite.current();
     WireCanvas* wc = NULL;
     do
     {
	  QDictIterator<WireCanvas> ite = ObjectSelect->getWireIterator();
	  wc = ite.current();
	  if (ite.current())
	  {
	       WireSelect = ite.current();
	       delWire();
	  }
     } while (wc);

     
     // Remove des wire des slots
     WireCanvas* wc2 = NULL;
     do
     {
	  QDictIterator<WireCanvas> ite2 = ObjectSelect->getWireIteratorSlot();
	  wc2 = ite2.current();
	  if (ite2.current())
	  {
	       WireSelect = ite2.current();
	       delWire();
	  }
     } while (wc2);
       
     ObjectSelect->clear();
     //On supprime l'objet de la feuille
     Sheet->delObjectCanvas(ObjectSelect->getObjectID());

     //Si l'objet n'est utilis par aucune feuille, il est remove du dictionnaire des metaobject
     int nbrs = 0;
     QDictIterator<SheetView> ito( dSheet );
     for ( ; ito.current(); ++ito )
	  if ( ito.current()->getObjectCanvas(ObjectSelect->getObjectID()) != 0 )
	       nbrs ++;
     if ( nbrs == 0 ) 
	  delete (ObjectSelect->getMetaObject());

     resetPointers();
     Canvas->update();
}


void MyWidget::addWireDial()
{
     if (Sheet == NULL) return;

     /* Gestion de l'ajout d'un connection entre object */
     // rcuprer l'objet source + son signal puis l'objet de destination + son slot. Enfin cr un metawire.
     // Ensuite en dduire les nodeitem correspondant

     // Pour savoir le nombre d'objet sur la feuille par le biais de la variable "nbr"
     int nbr=0;
     // On rcupre le nombre d'objet prsent sur la feuille
     Sheet->giveNbrObjInSheet(&nbr);
     // Donc si il y a au moins un objet sur la feuille nous pouvons continuer.
     if ( nbr > 0 ) {
	  /* Cration de la premiere QDialog pour selectionner la source et son signal (addwiredialog.cpp) */
	  // On utilisera la meme class "addWireDialog" pour selectionner le signal et le slot
	  // On mettra "true" pour dire que l'on veux selectionner un signal
	  SelectSignalDialog m(0, "addWireSignal", Sheet, ml);
	  /* Si un objet et un signal ont t choisient puis valids, nous ouvrons alors 
	      une seconde QDialog pour selectionner un slot donc en mettant "false" lors de la cration de notre addWireDialog */
	  if ( m.exec() ) {    
	       /* Cration de notre second addWireDialog pour la selection d'un slot */
	       SelectSlotDialog n(0, "addWireSignal", Sheet, ml, m.getMember());
	       // Si un slot est selectionn et valid on cr un metawire que l'on enverra pour cr un wirecanvas
	       if ( n.exec() ) {
		    Sheet->addWire(new MetaWire( m.getObjectName(), m.getMember(), n.getObjectName(), n.getMember()));
	       }
	  }
     }
     else 
	  QMessageBox::information( this, "To add a wire,","you should put an object on the sheet !");
}

void MyWidget::delWire()
{
     //cout << "In MyWidget::delWire() " << endl;
     if ( WireSelect != NULL ) {
	  WireCanvas * dwwc = WireSelect;
	  MetaWire *dwmw = dwwc->getMetaWire();
	  if ( dwmw != NULL ) {
/*	       ((ObjectCanvas*)(((NodeItem*)(dwwc->getToNode()))->getObjectCanvas()))->delWire(dwwc);
	       ((ObjectCanvas*)(((NodeItem*)(dwwc->getFromNode()))->getObjectCanvas()))->delWire(dwwc);*/

	       //on remove dans le dict dWireSlot de l'objet source en premier car il n'est pas en autodelete)       
	       NodeItem *nodeTo = (NodeItem*)(dwwc->getToNode());
	       ObjectCanvas *objectTo = ((ObjectCanvas*)(nodeTo->getObjectCanvas()));
	       objectTo->delWire(dwwc);
	       
	       // on remove dans le dict dWireSignal de l'objet de destination. Le WireCanvas sera delete en mme temps
	       NodeItem *nodeFrom = (NodeItem*)(dwwc->getFromNode());
	       ObjectCanvas *objectFrom = ((ObjectCanvas*)(nodeFrom->getObjectCanvas()));
	       objectFrom->delWire(dwwc); //dwmw->getSignal());

	       Sheet->delWire(dwwc);
	       
	       delete dwwc;
	       Canvas->update(); 
	  }
	  else 
	       cout<<"Del Wire MetaWire null"<<endl;
     }
     else 
     cout<<"DEl Wire WireSelect null"<<endl;
     
     WireSelect = NULL;
}

void MyWidget::orderWireDial()
{
     if (Sheet == NULL) return;

     orderDialog o( this, "orderWire",Sheet, false);  
     if ( o.exec() ) 
	  Sheet->setWireVector(o.getWireVector());
}


QString  MyWidget::getType(QString s)
{
     if ( s == "QObject*")
	  return "object";
     if ( s == "")
	  return "void";
     if ( s == "int")
	  return "int";
     if ( s == "float")
	  return "float";
     if ( s == "double" )
	  return "double";
     if ( s == "QString" )
	  return "string";
     if ( s == "bool" )
	  return "bool";
     return "unknown";

}



void MyWidget::addInitDial()
{
     int nbr=0;
     // On rcupre le nombre d'objet prsent sur la feuille
     if (Sheet)
	  Sheet->giveNbrObjInSheet(&nbr);

     // Donc si il y a au moins un objet sur la feuille nous pouvons continuer.
     if ( nbr > 0 ) {
	  SelectInitDialog o(this, "SelectInitDialog", Sheet, ml); 

	  // Si on appui sur "OK" sinon "Cancel"
	  if ( o.exec() ) {	
	       //ouvrir seconde fentre
	       QString slotInit = o.getMember();
	       InitValueDialog p(0, "addInitDialog2",slotInit);
	       if ( p.exec() ) {
 		    QString typeInit = (slotInit.section("(",1,1));
		    if (p.isDefine())
			 typeInit = "define";
		    else
			 typeInit = getType(typeInit.left( typeInit.length() - 1 ));
		    MetaInit *mi = new MetaInit(o.getObjectName(),slotInit,typeInit,p.getValue(),true); 
		    if (Sheet)
			 Sheet->addInit(mi, p.isPostInit());
	       }
	  }
     }
     else 
	  QMessageBox::information( this, "To add an initialisation,","you should add an object on the sheet !");
}

void MyWidget::delInit()
{
     InitCanvas* ic = InitSelect;

     //On recupre le metainit de l'initcanvas
     MetaInit *dimi = (InitSelect->getMetaInit());
     //On recupere l'objet correspondant
     ObjectCanvas *dioc = Sheet->getObjectCanvas(dimi->getObject());
     //on remove le initcanvas
     dioc->delInit(dimi->getSlot());
     if (Sheet)
	  Sheet->delInit(ic);
     //on met a jour le canvas.

     // robustesse accrue avec ces deux lignes.
     //dimi       = NULL; 
     //--> Ce metainit est il bien dtruit ?
     InitSelect = NULL; 

     Canvas->update();
}


void MyWidget::delIO()
{
     //IOCanvas* ic = IOSelect;
     
     MetaIO* dio = (IOSelect->getMetaIO());
     ObjectCanvas *dioc = Sheet->getObjectCanvas(dio->getObjectName());
     dioc->delIO(dio->getPublicName());
     
     IOSelect = NULL;
     Canvas->update();
}
     

void MyWidget::modifyInit()
{
     MetaInit *dimi = (InitSelect->getMetaInit());
     InitValueDialog p(0, "addInitDialog2",dimi->getSlot(), dimi->getValue(), InitSelect->isPostInit());
     if ( p.exec() ) {
	  QString typeInit = (dimi->getSlot().section("(",1,1));
	  typeInit = getType(typeInit.left( typeInit.length() - 1 ));
	  dimi->setValue(p.getValue());
	  if (InitSelect->isPostInit() !=  p.isPostInit())
	       if (Sheet)
		    Sheet->switchInit(InitSelect);
	  InitSelect->setPostInit(p.isPostInit());
     }
}


void MyWidget::orderInitDial(bool typeOfInit)
{
     if (Sheet == NULL) return;

     orderDialog o( this, "orderInit",Sheet,true,typeOfInit);  
     if ( o.exec() ) 
	  Sheet->setInitVector(o.getInitVector(),typeOfInit);
}



QRect MyWidget::computeCanvasBoundingBox(QCanvas* canvas)
{
     QRect cmax = canvas->rect();
     QRect res;

     res.setX(cmax.right());
     res.setY(cmax.bottom());
     res.setRight(cmax.left());
     res.setBottom(cmax.top());

     
     QCanvasItemList lst = canvas->allItems();
     cout << "computing bounding box" << endl;

     for (unsigned int i=0; i < lst.count() ; i++)
     {
	       QRect br = lst[i]->boundingRect();
	  
	       res.setX( (br.x() < res.x())?br.x():res.x());
	       res.setY( (br.y() < res.y())?br.y():res.y());
	       res.setRight ((br.right() > res.right())?br.right():res.right() );
	       res.setBottom((br.bottom() > res.bottom())? br.bottom(): res.bottom() );
	       cout << "Dimensions : " << res.x() << ", " << res.y() << "; " << res.width() << ", " << res.height() << 
		    " :: " << lst[i]->rtti() << endl;
     }


     return res;
}






void MyWidget::saveCanvas()
{
     /* Cration d'un QFileDialog pour dfinir un fichier de sauvegarde pour notre canvas */
     QFileDialog *fd = new QFileDialog(this,"Save the canvas");
     /* Configuration de ce QFileDialog */
     // QFileDialog de type "SAVE"
     fd->setMode( QFileDialog::AnyFile );
     // Dfinition des filtres pour le type de fichiers possibles
     fd->addFilter("PNG (*.png)");
     fd->addFilter("JPG (*.jpg)");
     fd->addFilter("BMP (*.bmp)");
     fd->addFilter("SVG (*.svg)");
     // Force la slction du filtre PNG (*.png)
     fd->setSelectedFilter(1);

     /* Execution du notre QFileDialog et regarde si on appui sur le bouton "save" */
     if ( fd->exec() == QDialog::Accepted ) {
	  // Rcupre le fichier slectionn
	  QString fn = fd->selectedFile(); 
	  // Rcupre le filtre slectionn
	  QString sf = fd->selectedFilter();
          // Extraction de l'extension du fichier
	  QString ext1 = sf.section("*",1,1); 
	  QString ext2 = ext1.left( ext1.length() - 1); 
          // Rajout de l'extension si l'utilisateur ne la pas mise	 
	  if ( fn.find( ext2 ) == -1 ) {
	       fn = fn + ext2; 
	  }
	  QRect rct = computeCanvasBoundingBox(Canvas);
	 
	  if ( ( sf == "PNG (*.png)")||( sf == "JPG (*.jpg)")||( sf == "BMP (*.bmp)") ) 
	  {
	       QPixmap pic(rct.width(),rct.height());
	       QPainter p(&pic);
	       Canvas->drawArea( rct, &p );
	       p.end();
	       pic.save( fn,(sf.section(" ",0,0)) );
	  }

	  if ( sf == "SVG (*.svg)" ) {
	       QPicture pic;
	       QPainter p( &pic );
	       Canvas->drawArea( rct, &p );
	       p.end();
	       pic.save( fn, "SVG" );
	  }
     }
}

void MyWidget::addLibrary()
{
     QString s = QFileDialog::getOpenFileName(
	  ".",
#ifdef WIN32
	  "Dll (*.dll)",
#else
	  "SO (*.so)",
#endif
	  this,
	  "open file dialog a Library",
	  "Choose a file to open" );
     
     bool isLib = false;
     if ( !s.isEmpty() ) {
	  QStringList lst = ml.getLibraries();
	  for (uint i = 0; i < lst.count(); i++ ) {
	       if ( lst[i] == s )
		    isLib = true;
	  }
	  if ( isLib )
	       QMessageBox::information( this, "Add a library","This library is already loaded !");
	  else
	  {
	       if (!ml.loadLibrary(s))
		    QMessageBox::information( this, "Ajout d'une librairie","Le chargement de la librairie a chou !");
	  }
     }
}


void MyWidget::addSheet()
{
     SheetDialog s( this, "AddSheet",true);
     connect( &s, SIGNAL(isSheet(QString,bool*)), this, SLOT(hasSheet(QString,bool*)));
     if ( s.exec() ) 
	  setNewSheet(s.getSheetName());
     s.disconnect( SIGNAL(isSheet(QString,bool*)));
}

void MyWidget::setNewSheet(QString ns)
{
     
     if ( view->isItemChecked(StateView) ) 
	  setMode(2);
     //else {
     if ( Sheet != 0 ) {
	  Sheet->hide();
	  Sheet->disconnect( SIGNAL(changeSheet(QString)));
	  view->setItemChecked ( indexSheet, false );
     } 
	  //}
     
     Sheet = new SheetView( this, ns );
     dSheet.insert(ns, Sheet);
     setCentralWidget(Sheet);
     Canvas = Sheet->canvas();
     Sheet->show();
     numIndex += 1;
     view->insertItem( ns ,numIndex);
     connect( Sheet, SIGNAL(changeSheet(QString)), this, SLOT(changeSelect(QString)));
     indexSheet = numIndex;
     view->setItemChecked ( indexSheet, true );
     viewActive->setText( "<font color=darkgreen size=3><b>"
			  "     ""View : </b>"+ns+"</font>");     
}


void MyWidget::delSheet()
{
     //on deselectionne l'item dans le menu qui correspond a la sheet
     view->setItemChecked ( indexSheet, false );
     //on cache la feuille et bascul en mode automate

     //on remove la feuille du menu
     view->removeItem( indexSheet ) ;
     //remove des objets et tous leurs composants

     QDictIterator<ObjectCanvas> ite = Sheet->getObjectIterator();
     for( ; ite.current(); ++ite ) {
	  ObjectSelect = ite.current();
	  delObject();
     }
     Sheet->disconnect( SIGNAL(changeSheet(QString)));
     //on supprime l'tat qui correspond avec toutes ses transitions
     State->delState(Sheet->name());
     //on remove la feuille du dictionnaire et dtruit la feuille en elle meme
     dSheet.remove(Sheet->name());
     Sheet = NULL;
     setMode(1);
     Canvas->update();
}


void MyWidget::createMetablockSheet(MetaBlock* mb0)
{
     BlockDialog bd(this);

     QDictIterator<SheetView> it(dSheet);

     for ( ; it.current(); ++it)
	  if (it.current()->isMacroblock())
	       bd.addItem(it.currentKey());
     
     if (bd.exec())
     {
	  QString s = bd.getName();  

	  Sheet = dSheet[s];
	  if (Sheet == NULL)
	  {
	       MetaBlock*  mbtmp = new MetaBlock(s);
	       mb0->linkToMetaBlock(mbtmp);
	       mbtmp->setFile(s);
	       MetaSheet* ms = importBlock(s);
	       setNewSheet(s);
	       Sheet->setAsMacroblock(true);
	       setMode(2);
	       if (ms != NULL)
	       {
		    importSheet(ms);
		    importIO();
		    importGraphicsBlock(s);
	       }

	       Sheet->canvas()->update();
	  }
	  else
	  {
	       //<!\todo rendre plus propre !
	       mb0->linkToMetaBlock((MetaBlock*)MetaObject::getObject(s));
	       changeSheet(s);
	  }
     }
}




void MyWidget::changeSheet(QString s)
{
     if (Sheet)
     {
	  Sheet->hide();
	  Sheet->disconnect( SIGNAL(changeSheet(QString)));
     }
     Sheet = dSheet[s];
     
     if (Sheet == NULL)
     {
	  MetaObject* mo = MetaObject::getObject(s);
	  if (mo != NULL)
	  {
	       if (mo->isBlock())
	       {
		    MetaBlock* mb = ((MetaBlock*)mo)->getMetaBlockLink();
		    if (mb == NULL)
		    {
			 // Ouvrir une bote de dialogue, etc ...
			 createMetablockSheet((MetaBlock*)mo);
			 return ; 
		    }
		    else 
			 Sheet = dSheet[mb->getID()];
	       }
	  }
     }
     
     if (Sheet)
     {
	  Sheet->show();
	  setCentralWidget(Sheet);
	  Canvas = Sheet->canvas();
	  connect( Sheet, SIGNAL(changeSheet(QString)), this, SLOT(changeSelect(QString)));
	  viewActive->setText( "<font color=darkgreen size=3><b>"
			  "     ""View : </b>"+s+"</font>");
	  setMode(2);
     }
}

void MyWidget::selectSheet(int sh)
{
     if ( sh > LastSheet )
	  if ( view->isItemChecked(StateView) )
	       setMode(2);
     
     view->setItemChecked ( indexSheet, false );
     indexSheet = sh;
     view->setItemChecked ( indexSheet, true );
     changeSheet(view->text(sh));
}

void MyWidget::changeSelect(QString ns)
{
     if ( view->isItemChecked(StateView) ) 
	  setMode(2);
 	  
     view->setItemChecked ( indexSheet, false );
     for ( uint i=0; i < view->count(); i++ )
     {
	  if ( view->text(i) == ns ) {
	       indexSheet = i;
	       view->setItemChecked ( indexSheet, true );
	       break;
	  }
     }
     changeSheet(ns);         
}

void MyWidget::renameSheet()
{
     SheetDialog r( this, "AddSheet",false,view->text(indexSheet));
     connect( &r, SIGNAL(isSheet(QString,bool*)), this, SLOT(hasSheet(QString,bool*)));
     if ( r.exec() ) 
     {
	  ChangeNameSheet(r.getSheetName());
     }
     r.disconnect(SIGNAL(isSheet(QString,bool*)));
}


void MyWidget::ChangeNameSheet(QString ns)
{
     SheetView* saveSheet = dSheet.take(view->text(indexSheet));
     State->renameStateID(view->text(indexSheet),ns);
     view->changeItem(indexSheet,ns);
     saveSheet->setName(ns);
     dSheet.insert(ns, saveSheet);
     viewActive->setText( "<font color=darkgreen size=3><b>"
			      "     ""View : </b>"+ns+"</font>");
}


void MyWidget::addStateDial()
{

     QStringList dSheetList;
     QDictIterator<SheetView> ite( dSheet );
     if ( ite.count() > 0 ) {
	  for( ; ite.current(); ++ite ) {
	       //cout<<State->isState(ite.currentKey())<<endl;
	       if ( !(State->isState(ite.currentKey())) ) 
	       {
		    if (!ite.current()->isMacroblock())
			 dSheetList.append(ite.currentKey());
	       }
	  }
	  if ( dSheetList.count() != 0 ) {
	       StateDialog s(0, "StateDialog", dSheetList);
	       // Si on appui sur "OK" sinon "Cancel"
	       if ( s.exec() ) 
		    setNewState(s.getStateName());
	  }
	  else 
	       QMessageBox::information( this, "Adding a state","You should create a new sheet to add states in the statemachine !");
     }
     else
	  QMessageBox::information( this, "Adding a state","You should create a new sheet to add states in the statemachine !");
}

 
void MyWidget::addTransitionDial()
{
// seulement s'il y a des tat presents sur l'automate


     QStringList dSheetList;
     QDictIterator<SheetView> ite( dSheet );
     //if ( ite.count() > 1 ) {
     if ( State->getNbrState()  > 1 ) {
	  for( ; ite.current(); ++ite ) {
	       if ( State->isState(ite.currentKey()) )
		    dSheetList.append(ite.currentKey());
	  }
	  TransitionDialog t(0, "addTransitionDialog", dSheetList, true);
	  connect( &t, SIGNAL(transitionPossible(QString,QString,bool*)), this, SLOT(transitionIsPossible(QString,QString,bool*)));
	  if ( t.exec() ) {
	       QString sourceEtat = t.getStateName();
	       dSheetList.clear();
	       QDictIterator<SheetView> ite( dSheet );
	       for( ; ite.current(); ++ite ) {
		    if ( ( State->isState(ite.currentKey()) )  && ( ite.currentKey() != sourceEtat ) )
			 dSheetList.append(ite.currentKey());
	       }
	       TransitionDialog tt(0, "addTransitionDialog", dSheetList, false, t.getToken()/*nameTransition*/);
	       if ( tt.exec() ) {
		    TransitionCanvas *tc = new TransitionCanvas(State->getStateCanvas(sourceEtat),State->getStateCanvas(tt.getStateName()/*destEtat*/), t.getToken()/*nameTransition*/);
		    State->addTransition(tc);
	       }
	  }
	  t.disconnect( SIGNAL(transitionPossible(QString,QString,bool*)));
     }
     else 
	  QMessageBox::information( this, "Adding a transition","There should be at least 2 states to add a transition !");
}

void MyWidget::delTransition()
{
//     TransitionSelect
     //cout<<"del transition "<< TransitionSelect->getName().ascii()<<endl;
     StateCanvas *StateFrom = (StateCanvas*)TransitionSelect->getSender();
     StateCanvas *StateTo = (StateCanvas*)TransitionSelect->getReceiver();

     StateTo->delTransitionReceiver(TransitionSelect);
     StateFrom->delTransitionSender(TransitionSelect);
     Canvas->update();
}


void MyWidget::setMode(int numMode)
{
// mode 1 = StateView
// mode 2 = SheetView
     switch( numMode )
     {
     case 1:
	  if ( Sheet != 0 ) {
	       Sheet->hide();
	       view->setItemChecked ( indexSheet, false );
	  }
	  menuSheet->setItemEnabled( DelSheet, false);
	  menuSheet->setItemEnabled( RenSheet, false); 
	  menuSheet->setItemEnabled( AddLib, false);
	  menuSheet->setItemEnabled( AddObj, false);
	  menuSheet->setItemEnabled( AddIni, false);
	  menuSheet->setItemEnabled( AddWir, false);
	  menuSheet->setItemEnabled( OrderPre, false);
	  menuSheet->setItemEnabled( OrderPost,false);
	  menuSheet->setItemEnabled( OrderWire,false);
	  menuSheet->setItemEnabled( AddMsi, false);
	  menuSheet->setItemEnabled( AddMsl, false);
	  menuState->setItemEnabled( AddSta, true);
	  menuState->setItemEnabled( AddTra, true);
	  view->setItemChecked ( StateView, true );
	  State->show();
	  setCentralWidget(State);
	  Canvas = State->canvas();
	  viewActive->setText( "<font color=red size=3><b>"
			       "    ""View : </b>State Machine</font>");	
	  break;
     case 2:
	  view->setItemChecked ( StateView, false );
	  State->hide();
	  menuSheet->setItemEnabled( DelSheet, true);
	  menuSheet->setItemEnabled( RenSheet, true);
	  menuSheet->setItemEnabled( AddLib, true);
	  menuSheet->setItemEnabled( AddObj, true);
	  menuSheet->setItemEnabled( AddIni, true);
	  menuSheet->setItemEnabled( AddWir, true); 
	  menuSheet->setItemEnabled( OrderPre, true);
	  menuSheet->setItemEnabled( OrderPost,true);
	  menuSheet->setItemEnabled( OrderWire,true);
	  if (Sheet != NULL)
	  {
	       if (Sheet->isMacroblock())
	       {
		    menuSheet->setItemEnabled( AddMsi, true);
		    menuSheet->setItemEnabled( AddMsl, true);
	       }
	  }
	  menuState->setItemEnabled( AddSta, false);
	  menuState->setItemEnabled( AddTra, false);
	  //view->setItemChecked ( indexSheet, true );	
	  break;
     }
}


void MyWidget::exportBlocks()
{
     QDictIterator<SheetView> itsv(dSheet);

     for ( ;  itsv.current() ; ++itsv)
     {
	  if (itsv.current()->isMacroblock())
	  {
	       ProjectWriter expg(true);
	       expg.addSheet(itsv.current());
	       expg.addApplication( QString(itsv.current()->name()).section(QDir::separator(),-1) );
	       if (!expg.writeFile(QString(itsv.current()->name()).section('.',0,-2) + ".xxx"))
		    cerr << "Failed to export graphics file : " << itsv.current()->name() << endl;
	  }
     }
}



void MyWidget::ExportDial()
{
     QDictIterator<StateCanvas> itsc = State->getStateIterator();
     if ( itsc.count() == 0 ) {
	  QMessageBox::information( this, "To save a project","Statemachine without any state !");
     }
     else {
	  if ( State->getInitialState().isEmpty() )
	       QMessageBox::information( this, "To save a project","You must define an initial state !");
	  else {
	       if ( dSheet.count() > itsc.count() ) {
		    QMessageBox q(  "To save a project",
				    "Some sheets are not represented on state machine,\n"
				    "During the project saving, they will be lost.\n"
				    "Do you want to continue ?",
				    QMessageBox::Question,  
				    QMessageBox::Yes | QMessageBox::Default,
				    QMessageBox::No  | QMessageBox::Escape,
				    QMessageBox::NoButton  ); 
		    if ( q.exec() == QMessageBox::Yes )   
			 Export();
	       }
	       else
		    Export();
	  }
     }
      
}

void MyWidget::Export()
{

//mettre dans meta sheet ce qu'il faut mettre(token pre connexion et post)
//parcourir chaque shhet du dict dSheet
     MetaSheet::clear();


     QDictIterator<SheetView> ite( dSheet );
     if ( State->getNbrState()  > 0 ) {
	  for( ; ite.current(); ++ite ) { 
	       SheetView *sv = ite.current();
	       //on cre une nouvelle metasheet
	       MetaSheet *ms = new MetaSheet(ite.currentKey());
	       if (sv->isMacroblock())
		    ms->setLinkedToBlock();

               //on dfinit le tokensender
	       ms->setTokenSender(sv->getTokenSender());
	       //on definit les pre/ post-connexions
	       if (ms != NULL)
	       {
		    sv->setInit(ms);
		    sv->setWire(ms);
	       }
	  }
     }

     MetaConfigWriter *expa = new MetaConfigWriter();
     expa->addLibraries(&ml);
     expa->addDefines();
     expa->addObjects();
     expa->addSheets();
     if (!State->getInitialState().isEmpty())
	  expa->setInitialState(State->getInitialState());
     if (!State->getTerminalState().isEmpty())
	  expa->setTerminalState(State->getTerminalState());
     State->setTransition(expa);         

     ProjectWriter *expg = new ProjectWriter();
     //expg->addApplication("application.xml");
     expg->addSheets(State->getStateIterator(),dSheet);


     QFileDialog *fd = new QFileDialog(this,"Save the canvas");
     /* Configuration de ce QFileDialog */
     // QFileDialog de type "SAVE"
     fd->setMode( QFileDialog::AnyFile );
     // Dfinition des filtres pour le type de fichiers possibles
     fd->addFilter("XXX (*.xxx)");
     // Force la slction du filtre PNG (*.png)
     
     QString fn;
     QString fn2;
     QString fn3;
     /* Execution du notre QFileDialog et regarde si on appui sur le bouton "save" */
     if ( fd->exec() == QDialog::Accepted ) {
	  // Rcupre le fichier slectionn
	  fn = fd->selectedFile(); 
	  // Rcupre le filtre slectionn
	  //QString sf = fd->selectedFilter();
          // Extraction de l'extension du fichier
	  //QString ext1 = sf.section("*",1,1); 
	  //QString ext2 = ext1.left( ext1.length() - 1); 
          // Rajout de l'extension si l'utilisateur ne la pas mise	 
	  if ( fn.right(4) != ".xxx" ) {
	       fn2 = fn + ".xml";
	       fn = fn + ".xxx";

	  }
	  else 
	       fn2 = fn.left( fn.length() - 4) + ".xml";
	  
	  fn3 = fn2.right(fn2.length() - fn2.findRev(QDir::separator()) - 1);
	  //std::cerr << "Nom du fichier  exporter : " << fn3.ascii() << std::endl;

	  expg->addApplication(fn3);
	  if ( expa->writeFile(fn2) ) 
	  {
	       if (! expg->writeFile(fn) ) 
		    cerr<<"Erreur lors de l'exportation graphique"<<endl;  
	       else
		    exportBlocks();
	  }
	  else
	       cerr<<"Erreur lors de l'exportation de l'application"<<endl;
     }
     saveName = fn;

     delete expa;
     delete expg;
     
}



void MyWidget::autoPlaceObject(ObjectCanvas* oc, int idx)
{
     int xoffset = 40;
     int yoffset = 40;
     int xgrid = 5;
     int ygrid = 10;

     int width = oc->canvas()->width();
     int height = oc->canvas()->height();
     
     int igrid = idx%xgrid * width/xgrid; 
     int jgrid = idx/xgrid * height/ygrid;

     oc->move(igrid+xoffset,jgrid+yoffset);
}


void MyWidget::autoPlaceState(StateCanvas* sc, int idx)
{
     int xoffset = 100;
     int yoffset = 40;
     int xgrid = 5;
     int ygrid = 10;

     int width = sc->canvas()->width();
     int height = sc->canvas()->height();
     
     int igrid = idx/ygrid * width/xgrid; 
     int jgrid = idx%ygrid * height/ygrid;

     sc->move(igrid+xoffset,jgrid+yoffset);
}



ObjectCanvas* MyWidget::checkObject(QString objname,int& idx)
{
     ObjectCanvas* oc = NULL;
     MetaObject* mo = MetaObject::getObject(objname);
     if (mo != NULL)
     {
	  oc = Sheet->getObjectCanvas(objname);
	  if (oc == NULL)
	  {
	       Sheet->addObject(mo);
	       oc = Sheet->getObjectCanvas(objname);
	       autoPlaceObject(oc,idx);
	       idx++;
	  }
     }
     return oc;
}


void MyWidget::importIO()
{
     // traitement des slots 
     int idx = 0;
     QDictIterator<MetaIO> ioslt = MetaBlock::getSlotList();

     for ( ; ioslt.current() ; ++ioslt)
     {
	  QString key = ioslt.currentKey();
	  QString sheetName = key.left(key.find("::"));
	  changeSelect(sheetName);
	  ObjectCanvas* oc = checkObject(ioslt.current()->getObjectName(),idx);
	  if (!oc->hasIO(ioslt.currentKey()))
	       Sheet->addIO(ioslt.current(), true);
     }


     // traitement des signaux
     QDictIterator<MetaIO> iosig = MetaBlock::getSignalList();
     
     for ( ; iosig.current() ; ++iosig)
     {
	  QString key = iosig.currentKey();
	  QString sheetName = key.left(key.find("::"));
	  changeSelect(sheetName);
	  ObjectCanvas* oc = checkObject(iosig.current()->getObjectName(),idx);
	  if (!oc->hasIO(iosig.currentKey()))
	       Sheet->addIO(iosig.current(), false);
     }

}


MetaSheet* MyWidget::importBlock(QString file)
{
     MetaConfigReader impa; 
     impa.setEditMode(true);
     if ( impa.openFile(file,file) )
     {
	  if ( !impa.parseLibrariesSection() )
	  {
	       cerr << "Failed to parse libraries section." << endl;
	       return NULL ;
	  }
	  else 
	  {
	       QStringList lib  = impa.getLibraryNames();
	       for ( QStringList::Iterator it = lib.begin(); it != lib.end(); ++it ) 
	       {
		    QString l = *it;
		    ml.loadLibrary(l);
	       }
	  }
	  if (!impa.parseObjectsSection())	 
	  {
	       cerr << "Failed to parse objects section." << endl;
	       return NULL;
	  }
	  if (!impa.parseSignalsSection())	 
	  {
	       cerr << "Failed to parse signals section." << endl;
	       return NULL;
	  }
	  if (!impa.parseSlotsSection())	 
	  {
	       cerr << "Failed to parse slots section." << endl;
	       return NULL;
	  }

	  if (!impa.parseDefineSection())
	  {
	       cerr << "Failed to parse defines section." << endl;
	  }

	  MetaSheet* ms = impa.parseSheetSection(file);
	  if (ms != NULL)
	       ms->setLinkedToBlock();
	  return ms;
     }
     else return NULL;
}

void MyWidget::importGraphicsSheet(ProjectReader* prd)
{
     int tmpIdx=0;

     // Boucle d'import des objets
     ObjectPlacement op = prd->nextObject();
     while (op.validity)
     {
	  ObjectCanvas *oc = checkObject(op.id,tmpIdx);  //->Sheet->getObjectCanvas(op.id);
	  oc->move( op.x, op.y);
	  op = prd->nextObject();
     }
     
     // Boucle d'import des liaisons
     WirePlacement wp = prd->nextWire();
     while(wp.validity)
     {
	  WireCanvas *wc = NULL;
	  
 	  QDictIterator<WireCanvas> itwc = (Sheet->getObjectCanvas(wp.objSender))->getWireIterator();
	  for ( ; itwc.current() ; ++itwc)
	  {
	       if ( ( (itwc.current()->getMetaWire())->getSender() == wp.objSender )
		    && ( (itwc.current()->getMetaWire())->getReciever() == wp.objReciever )
		    && ( (itwc.current()->getMetaWire())->getSignal() == wp.signal )
		    && ( (itwc.current()->getMetaWire())->getSlot() == wp.slot ) )
	       {
		    wc = itwc.current();
		    break;
	       }
	  }
	  if ( wc == NULL )
	       cerr<<"Aucun WireCanvas correspondant"<<endl;
	  else 
	  {
	       wc->setVisibleGrip(true);			 
	       int xtp, ytp;
	       wc->getToPoint(xtp,ytp);
	       for ( int k=0; k < 4; k++ ) // modif.
	       {  
		    double xgtp = wp.pos[k][0];
		    double ygtp = wp.pos[k][1];
		    if (k == 3)
			 if ( (int)ygtp != (ytp-3) )
			      ygtp = ytp-3;
		    (wc->getGripPoint(k))->savePoint(xgtp,ygtp);
	       }
	       (wc->getGripPoint(3))->setLine();
	       wc->setVisibleGrip(false);
	       wc->canvas()->update();
	  }

	  wp = prd->nextWire();
     }

     // on replace les grips
     QCanvasItemList cli = (Sheet->canvas())->allItems();
     for ( uint i = 0; i < cli.count(); i++)
	  if ( (cli[i])->rtti() == RTTI_GRIPWIRE )
	       ((GripWire*)cli[i])->setPoint();
}




void MyWidget::importGraphicsBlock(QString name)
{
     MetaObject* mo = MetaObject::getObject(name);
     if (mo == NULL) 
	  return;
     
     if (mo->isBlock())
     {
	  MetaBlock* mb = (MetaBlock*)mo;
	  QString filename = mb->getFile();
	  
	  ProjectReader prd;
	  prd.setNameSpace(name);
	  if (prd.openFile(filename.section('.',0,-2) + ".xxx"))
	  {
	       ObjectPlacement sp= prd.nextSheet();
//	       cout << "Sheet loaded has name '" << name.ascii() << "' and id '" << sp.id.ascii() << "'" << endl;
	       //if ( sp.validity )
	       //{
	       changeSelect(name);
	       importGraphicsSheet(&prd);
		    //}
	  }
     }
}


bool MyWidget::importApplication(QString file)
{
     // Chargement du fichier de configuration d'une application
     MetaConfigReader impa; 
     impa.setEditMode(true);
     if ( impa.openFile(file) )
     {
	  if ( !impa.parseLibrariesSection() )
	  {
	       cerr << "Failed to parse libraries section." << endl;
	       return false ;
	  }
	  else 
	  {
	       QStringList lib  = impa.getLibraryNames();
	       for ( QStringList::Iterator it = lib.begin(); it != lib.end(); ++it ) 
	       {
		    QString l = *it;
		    ml.loadLibrary(l);
	       }
	  }
	  if (!impa.parseObjectsSection())	 
	  {
	       cerr << "Failed to parse objects section." << endl;
	       return false;
	  }

	  if (!impa.parseSheetsSection())
	  {
	       cerr << "Failed to parse sheets section." << endl;
	       return false;
	  }
	  if (!impa.configureStateMachine(&msm))
	  {
	       cerr << "Failed to parse state machine section." << endl;
	       return false;
	  }
	  if (!impa.parseDefineSection())
	  {
	       cerr << "Failed to parse defines section." << endl;
	  }
	  return true;
     }
     else return false;
}



void MyWidget::importSheet(MetaSheet* ms)
{
	  int i;
	  int occ = 0;
	  for (i=0; i < ms->getPreConnectionsNumber() ; i++)
	  {
	       MetaInit mi = ms->getPreConnection(i);
	       // get object and create it.
	       checkObject(mi.getObject(),occ);
	       MetaInit *pmi = new MetaInit(mi);
	       Sheet->addInit(pmi,false); // ajout de la post connection 
	  } 
	  for (i=0; i < ms->getPostConnectionsNumber() ; i++)
	  {
	       MetaInit mi = ms->getPostConnection(i);
	       // get object and create it.
	       checkObject(mi.getObject(),occ);
	       MetaInit *pmi = new MetaInit(mi);
	       Sheet->addInit(pmi,true); // ajout de la post connection 
	  } 
	  for (i=0; i < ms->getConnectionsNumber() ; i++)
	  {
	       MetaWire mw = ms->getConnection(i);
	       checkObject(mw.getSender(),occ);
	       checkObject(mw.getReciever(),occ);
	       MetaWire* pmw=new MetaWire(mw);
	       Sheet->addWire(pmw);
	  }

	  // Traitement de la machine  tat (tokensender)
	  if ( ! ms->getTokenSender().isEmpty() ) 
	  {
	       checkObject( ms->getTokenSender(), occ);
	       Sheet->setTokenSender( ms->getTokenSender() );
	  }
}



void MyWidget::importStateMachine()
{
     // Sparation pour traiter les statecanvas en passe n2 
     QDictIterator<MetaSheet> it = MetaSheet::getIterator();
     // A raliser tout  la fin ?
     it = MetaSheet::getIterator();
     for ( ; it.current() ; ++it)
     {
	  setMode(1); // passage en mode machine  tat
	  MetaTokenAlternative* mta = msm.getTransitionsStartingFrom(it.current()->getSheetID());
	  if ( mta != NULL ) {
	       MetaTokenAlternative::Iterator itm;
	       for ( itm = mta->begin(); itm != mta->end(); ++itm ) 
		    State->addTransition(new TransitionCanvas(State->getStateCanvas(it.current()->getSheetID()),State->getStateCanvas(itm.data()),itm.key()));
	  }
     }

     // Traitement tats initiaux et finaux de la machine  tats
     if (! msm.getTerminalState().isEmpty()) {
	  if ( (msm.getTerminalState() == "end") && (dSheet["end"]) )
	       State->setTerminalState( msm.getTerminalState() );
	  else 
	       if (msm.getTerminalState() != "end")
		    State->setTerminalState( msm.getTerminalState() );
     }

     msm.start();
     if (! msm.getCurrentState().isEmpty())
	  State->setInitialState( msm.getCurrentState() );
}

void MyWidget::importElements()
{
     // Creation de ce qu'il faut ...
     // Commenons par les feuilles !
     QDictIterator<MetaSheet> it = MetaSheet::getIterator();
     int idxsheet = 0;
     for ( ; it.current() ; ++it)
     {
	  // creation de la feuille
	  setNewSheet(it.currentKey());
	  setMode(1); // passage en mode machine  tat
	  if ( !(it.current()->isLinkedToBlock()))	       
	  {
	       setNewState(it.currentKey()); // creation du nouvel tat
	       StateCanvas *sc = State->getStateCanvas(it.currentKey()); 
	       autoPlaceState(sc,idxsheet);
	  }
	  else
	  {
	       Sheet->setAsMacroblock(true);
	  }
	       
	  changeSelect(it.currentKey()); // Se mettre dans la bonne feuille.
	  importSheet(it.current());
	  	  
	  idxsheet ++;
     }

     importIO();

}


void MyWidget::Import(QString fileName)
{
     if (fileName.isEmpty())
     {
	  fileName = QFileDialog::getOpenFileName(
	       ".",
	       "Project files (*.xxx);;Application files (*.xml)",
	       this,
	       "open a file project",
	       "Choose a file project to open" );
     }

     if (fileName.isEmpty())
	  return;

     saveName = fileName ; 

     if ( fileName.right(3) == "xml")
     {
	  clear();
	  if (!importApplication(fileName))
	  {
	       QMessageBox::critical( this, "Import error","Failed to import application file.");
	       clear();
	       return ;
	  }
	  importElements();
	  importStateMachine();
	  setMode(1);
	  Canvas->update();  
	  return;
     }

     ProjectReader prd;

     if (!prd.openFile(fileName))
     {
	  cerr << "Can't open project file " << fileName.ascii() << endl;
	  return ;
     }
     clear();

     
     QString applicationFileName = prd.getApplicationFilename();
     QDir dir(applicationFileName);
     if (dir.isRelative())
	  applicationFileName = fileName.left( fileName.findRev(QDir::separator())) + QDir::separator() + applicationFileName;


     if (!importApplication(applicationFileName))
     {
	  QMessageBox::critical( this, "Import error","Failed to import application file.");
	  clear();
	  return;
     }

     importElements();

     // Boucle d'import graphique
     ObjectPlacement sp = prd.nextSheet();
     while(sp.validity)
     {
	  setMode(1);
	  StateCanvas *sc = State->getStateCanvas(sp.id);
	  sc->move(sp.x,sp.y);
	  changeSelect(sp.id);
	  importGraphicsSheet(&prd);
	  sp = prd.nextSheet();
     }

     // Boucle d'import graphique des macro-blocs
     QDictIterator<SheetView> itsc(dSheet);
     for ( ; itsc.current() ; ++itsc)
	  if (itsc.current()->isMacroblock())
	       importGraphicsBlock(itsc.currentKey());

     //on se replace sur la statemachineview
     setMode(1);
     importStateMachine();
     Canvas->update();  
}


void MyWidget::clear()
{
     //Dtruire les lments du menu
     view->clear();

     view->insertItem( "StateMachine" , this, SLOT(StateMachine()),0, StateView);
     view->insertSeparator();
     QLabel *caption = new QLabel( "<font color=blue size=3><u><b>"
				   "sheet(s)</b></u></font>", view );
     caption->setAlignment( Qt::AlignCenter );
     numIndex = LastSheet;
     view->insertItem( caption , numIndex );
     numIndex += 1;
     indexSheet = numIndex;
     //basculement en mode statemachine
     setMode(1);
     
     MetaObject::clear();
     MetaSheet::clear();
     dSheet.clear();
     Sheet = NULL;
     //pour l'automate
     
     State->clear();

     //remove des lment de la metastatemachine
     msm.clear();
     Canvas->update();  
}


void MyWidget::zoom(int z)
{
     QCanvasView* zoomCanvas = NULL;

     if (! State->isHidden())
	  zoomCanvas = State;
     else
	  if (Sheet !=  NULL)
	  {
	       if (! Sheet->isHidden())
		    zoomCanvas = Sheet;
	  }


     if (zoomCanvas == NULL)
	  return;

     QWMatrix matrix = zoomCanvas->worldMatrix();
     switch ( z)
     {
     case ZoomIn :
	  matrix.scale(2.0f,2.0f); //in
	  break;
     case ZoomOut :
	  matrix.scale(0.5f,0.5f); //out
	  break;
     default :
	  matrix.reset(); //11
	  break;
     }

     zoomCanvas->setWorldMatrix(matrix);
}

void MyWidget::run()
{
     if ( saveName.isEmpty())
     {
	  ExportDial();
     }
     else
     {
	  switch( QMessageBox::information( this, "","Check if you saved project now.\n"
					    "You can do it by pushing button \"save\"",
					    "&Save", "&Don't save", "Cancel",0 ,2 ) ) 
	  { 
	  case 0: 
	       ExportDial();
	       break;
	  case 1: 
	       break;
	  case 2: 
	       return;
	  }
     }
     

     if (!saveName.isEmpty())
     {
	  ExecDialog ed(this, "toto", saveName);
	  ed.exec();
     }
}


void MyWidget::addMetaSlot()
{
     if (Sheet == NULL)
	  return;

     SelectSlotDialog ssd(0,"Metaslot selection",Sheet, ml);
     if (ssd.exec())
     {
	  bool ok; 
	  QString txt = QInputDialog::getText("Meta Slot Name",
					      "Please enter the slot name\nwithout any parameter nor parenthesis:",
					      QLineEdit::Normal, QString::null, &ok, this);
	  if (ok)
	  {
	       txt = Sheet->name()+ QString("::") + txt + (ssd.getMember()).section('(',1,1,QString::SectionIncludeLeadingSep);
	       MetaIO* mio = new MetaIO(txt, ssd.getObjectName(), ssd.getMember());
	       Sheet->addIO(mio,true);
	       MetaObject* mo = MetaObject::getObject(Sheet->name());
	       if (mo->isBlock())
		    ((MetaBlock*)mo)->addSlot(mio);
	  }
     }
}


void MyWidget::addMetaSignal()
{
     if (Sheet == NULL) 
	  return;

     SelectSignalDialog ssd(0,"Metasignal selection",Sheet, ml);
     if (ssd.exec())
     {
	  bool ok; 
	  QString txt = QInputDialog::getText("Meta Signal Name",
					      "Please enter the signal name\nwithout any parameter nor parenthesis:",
					      QLineEdit::Normal, QString::null, &ok, this);
	  if (ok)
	  {
	       txt = Sheet->name() + QString("::") + txt + (ssd.getMember()).section('(',1,1,QString::SectionIncludeLeadingSep);
	       MetaIO* mio = new MetaIO(txt, ssd.getObjectName(), ssd.getMember());
	       Sheet->addIO(mio,false);
	       MetaObject* mo = MetaObject::getObject(Sheet->name());
	       if (mo->isBlock())
		    ((MetaBlock*)mo)->addSignal(mio);
	  }
     }
}


void MyWidget::refresh()
{
     QCanvasView* refreshCanvas = NULL;

     if (! State->isHidden())
	  refreshCanvas = State;
     else
	  if (Sheet !=  NULL)
	  {
	       if (! Sheet->isHidden())
		    refreshCanvas = Sheet;
	  }

     if ( refreshCanvas == NULL )
	  return ;


     QCanvasItemList itemList =  refreshCanvas->canvas()->allItems();

     
     for( uint i = 0; i < itemList.count(); i++)
     {
	  switch( itemList[i]->rtti() )
	  {
	  case QCanvasItem::Rtti_Text :
	       itemList[i]->setZ(DEPTH_CANVASTEXT);
	       break;
	  case QCanvasItem::Rtti_Ellipse:
	       itemList[i]->setZ(DEPTH_CANVASELLIPSE);
	       break;
	  case RTTI_STATECANVAS:
	       itemList[i]->setZ(DEPTH_STATECANVAS);
	       break;
	  case RTTI_INITCANVAS:
	       itemList[i]->setZ(DEPTH_INITCANVAS);
	       break;
	  case QCanvasItem::Rtti_Polygon:
	       itemList[i]->setZ(DEPTH_CANVASPOLYGON);
	       break;
	  case RTTI_IOCANVAS:
	       itemList[i]->setZ(DEPTH_IOCANVAS);
	       break;
	  case RTTI_NODEITEM:
	       itemList[i]->setZ(DEPTH_NODEITEM);
	       break;
	  case QCanvasItem::Rtti_Line:
	       itemList[i]->setZ(DEPTH_CANVASLINE);
	       break;
	  case RTTI_LINEWIRE:
	       itemList[i]->setZ(DEPTH_LINEWIRE);
	       break;
	  case RTTI_TRANSITIONCANVAS:
	       itemList[i]->setZ(DEPTH_TRANSITIONCANVAS);
	       break;
	  case RTTI_WIRECANVAS:
	       itemList[i]->setZ(DEPTH_WIRECANVAS);
	       break;
	  case RTTI_GRIPWIRE:
	       itemList[i]->setZ(DEPTH_GRIPWIRE);
	       break;
	  case QCanvasItem::Rtti_Rectangle:
	       itemList[i]->setZ(DEPTH_CANVASRECTANGLE);
	       break;
	  case RTTI_OBJECTCANVAS:
	       itemList[i]->setZ(DEPTH_OBJECTCANVAS);
	       break;
	  default:
	       itemList[i]->setZ(100);
	       break;
	  }
     }
}
