/*
  name: projectreader.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include "projectreader.h"

#include <qfile.h>
#include <iostream>


ObjectPlacement ProjectReader::op_init = { false, 0, 0.0, 0.0 };
WirePlacement ProjectReader::wp_init = { false, 0, 0, 0, 0, {{0,0},{0,0},{0,0},{0,0}}}; 


bool ProjectReader::openFile(QString s)
{
     QFile file(s);
     if (file.open(IO_ReadOnly))
     {
	  if (!graphique.setContent(&file))
	  {
	       std::cerr << "[ProjectReader] Couldn't parse project file." << std::endl;
	       file.close();
	       return false;
	  }
     }
     else
     {
	  std::cerr << "[ProjectReader] Couldn't open this file." << std::endl;
	  file.close();
	  return false;
     }

     QDomElement root = graphique.documentElement();

     QDomNodeList dnl = root.elementsByTagName("application");
     if (dnl.count() > 0)
     {
	  QDomElement application = dnl.item(0).toElement();
	  if (application .hasAttribute("fileApplication"))
	       filename = application.attribute("fileApplication");
	  else
	  {
	       if (application.hasAttribute("fileBlock"))
	       {
		    filename = application.attribute("fileBlock");
		    block = true;
	       }
	       else
	       {
		    std::cerr << "[ProjectReader] 'application' markup without attribute 'fileApplication'." << std::endl;
		    return false;
	       }
	  }
     }
     else
     {
	  std::cerr << "[ProjectReader] Project file '" << s.ascii()
		    << "' without any associated application." << std::endl;
	  return false;
     }



     QDomElement sheets;


     QDomNodeList dnl2 = root.elementsByTagName("sheets");
     if (dnl2.count() > 0)
	  sheets = dnl2.item(0).toElement();
     else
     {
	  std::cerr << "[ProjectReader] Project file without any sheet data." << std::endl;
	  return false;
     }

     sheetsList = sheets.elementsByTagName("sheet");
     sheetsIdx = 0;

     return true;
}


ObjectPlacement ProjectReader::nextSheet()
{
     if (sheetsIdx >= sheetsList.count())
	  return op_init;

     QDomElement sheet = sheetsList.item(sheetsIdx).toElement();
     ObjectPlacement res = op_init;

     if (sheet.hasAttribute("id"))
	  res.id = block ? nameSpace + "::" : sheet.attribute("id");
     if (sheet.hasAttribute("x"))
	  res.x = sheet.attribute("x").toDouble();
     if (sheet.hasAttribute("y"))
	  res.y = sheet.attribute("y").toDouble();
     sheetsIdx++;

     QDomNodeList dnl = sheet.elementsByTagName("objects");
     if (dnl.count() > 0)
     {
	  objectsList = dnl.item(0).toElement().elementsByTagName("object");
	  objectsIdx = 0;
     }
     else 
	  return res;

     QDomNodeList dnl2 =  sheet.elementsByTagName("wires");
     if (dnl2.count()> 0)
     {
	  wiresList = dnl2.item(0).toElement().elementsByTagName("wire");
	  wiresIdx = 0;
     }
     else
	  return res;
     

     res.validity = !res.id.isEmpty();

     return res;
}


ObjectPlacement ProjectReader::nextObject()
{
     if (objectsIdx >= objectsList.count() )
	  return op_init;

     QDomElement object = objectsList.item(objectsIdx).toElement();

     ObjectPlacement res = op_init;

     if (object.hasAttribute("id"))
	  res.id = ((block)? nameSpace + "::" : QString()) + object.attribute("id");
     if (object.hasAttribute("x"))
	  res.x = object.attribute("x").toDouble();
     if (object.hasAttribute("y"))
	  res.y = object.attribute("y").toDouble();
     
     objectsIdx++;
     res.validity = !(res.id.isEmpty());
     
     return res;
}

WirePlacement ProjectReader::nextWire()
{
     if (wiresIdx >= wiresList.count() )
	  return wp_init;
     
     QDomElement wire = wiresList.item(wiresIdx).toElement();

     WirePlacement res = wp_init;

     if (wire.hasAttribute("objsource"))
	  res.objSender = (block ? nameSpace + "::": QString()) + wire.attribute("objsource");
     if (wire.hasAttribute("objdest"))
	  res.objReciever = (block ? nameSpace + "::" : QString()) + wire.attribute("objdest");
     if (wire.hasAttribute("signal"))
	  res.signal = wire.attribute("signal");
     if (wire.hasAttribute("slot"))
	  res.slot = wire.attribute("slot");
     
     if (!wire.hasChildNodes())
	  return res;

     QDomNodeList grips = wire.childNodes();
     if ( grips.count() != 4)
	  return res;
     
     for (unsigned int i=0; i < 4; i++)
     {
	  QDomElement grip = grips.item(i).toElement();
	  if (grip.hasAttribute("x"))
	       res.pos[i][0] = grip.attribute("x").toDouble();
	  if (grip.hasAttribute("y"))
	       res.pos[i][1] = grip.attribute("y").toDouble();
     }


     wiresIdx ++;
     res.validity = true;

     return res;
}
