/*
  name: metalibrary.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include <metalibrary/metalibrary.h>
#include <qmetaobject.h>
#include <qdir.h>
#include <iostream>
#include <qregexp.h>

MetaLibrary::MetaLibrary()
{
     //    libraryMap.setAutoDelete(true);
// -> "tait source de problemes.
}


MetaLibrary::~MetaLibrary()
{

}


MetaLibrary::MetaLibrary(const MetaLibrary & lib)
{
     objectMap=lib.objectMap;
     libraryMap=lib.libraryMap;
}

QString MetaLibrary::genericLibName(QString s)
{
	QDir d(s);
	s = QDir::cleanDirPath(d.absPath());
	
	QFileInfo finf(s);
	QString fn = finf.fileName();

	QRegExp unixRE("^lib(\\w+)\\.so$");
	QRegExp winRE("^(\\w+)\\.dll$");
	QString finalFileName;

	if ( unixRE.exactMatch(fn))
		finalFileName = unixRE.cap(1);
	//else
	//	std::cout << "No unix match" << std::endl;

	if ( winRE.exactMatch(fn))
		finalFileName = winRE.cap(1);
	//else
	//	std::cout << "No windows match" << std::endl;

	if (finalFileName.isEmpty())
		return s;

	//std::cout << "Captured filename " << finalFileName.ascii() << "!" << std::endl;

	finf.setFile(finf.dirPath(), finalFileName);
	//std::cout << "Generic name " << finf.absFilePath().ascii() << std::endl;

	return finf.absFilePath();
	//return QString::null;
}

bool MetaLibrary::loadLibrary(QString s)
{
	//QDir d(s);
	//s = QDir::cleanDirPath(d.absPath());
	

    s = genericLibName(s);

     if (libraryMap[s] != NULL)
	    return true;

     QLibrary* lib = new QLibrary(s);
     
     if (!lib->load())
     {
	  std::cerr << "[Meta] Failed to load library probably missing links." << std::endl;
	  return false;
     }

     //libraryMap[s] = lib;
     libraryMap.insert(s,lib);

     MetaLibraryFill mlf = (MetaLibraryFill)lib->resolve("metalib_fill");
     if (mlf == NULL)
     {
	     std::cerr << "[META] Cannot find symbol metalib_fill (missing 'lib.cpp' ?)." << std::endl;
	     return false;
     }
     

     mlf(&objectMap);
     return true;
}

QStringList MetaLibrary::getObjects()
{
     QStringList l;

     QDictIterator<MetaWrapper> it(objectMap);
     for (; it.current(); ++it)
     {
	  l.append(it.currentKey());
     }
     return l;
}


QStringList MetaLibrary::getLibraries()
{
     QStringList l;

     QDictIterator<QLibrary> it( libraryMap);
     for( ; it.current(); ++it )
	  l.append(it.currentKey());

     return l;
}


QObject* MetaLibrary::createObject(QString objName, QObject* parent, const char* name)
{
     MetaWrapper* mw = objectMap.find(objName);
     if (mw == NULL)
	  return NULL;

#ifdef DEBUG
     std::cout  << "instanciating object " << objName.ascii() << std::endl;
#endif
     QObject* obj = mw->createObject(parent,name);
     return obj;
}


void MetaLibrary::destroyObject(QObject* obj, QString objName)
{
     MetaWrapper* mw = objectMap.find(objName);
     if (mw == NULL)
     {
	  if (obj != NULL)
	       delete obj;
	  return;
     }

     mw->destroyObject(obj);
}

QStrList MetaLibrary::getObjectSignals(QObject* obj, bool super)
{
     if (obj == NULL)
	  return QStrList();

#if QT_VERSION > 0x03ffff
     QStrList list;
     const QMetaObject * mo = obj->metaObject();
     for ( int i = 0 ; i < mo->methodCount(); i++ ) {
	 const QMetaMethod mm = mo->method(i);
	 if ( mm.methodType() != QMetaMethod::Signal )
	    list.append( QString( mm.signature() ) );
     }
     if( super ) {
	 const QMetaObject * moSuper = mo->superClass();
	 if( moSuper ) {
	     QStrList listSuper = getObjectSignals( QString(moSuper->className()),super);
	     for( unsigned int i = 0 ; i < listSuper.count() ; i++ )
		     list.append(  listSuper.at(i) );
	 }
     }
     return list;
#else
     return obj->metaObject()->signalNames(super);
#endif
}

// Sera probablement source de pb.
QStrList MetaLibrary::getObjectSignals(QString objName, bool super)
{
     QObject* obj = createObject(objName);
     if (obj == NULL)
	  std::cerr << "Required class " << objName.ascii() << " not found." << std::endl;
     QStrList l = getObjectSignals(obj, super);
     //delete obj;
     destroyObject(obj, objName); 
     return l;
}


QStrList MetaLibrary::getObjectSlots(QObject* obj, bool super)
{
     if (obj == NULL)
	  return QStrList();

#if QT_VERSION > 0x03ffff
     QStrList list;
     const QMetaObject * mo = obj->metaObject();
     for ( int i = 0 ; i < mo->methodCount(); i++ ) {
	 const QMetaMethod mm = mo->method(i);
	 if ( mm.methodType() != QMetaMethod::Slot )
	    list.append( QString( mm.signature() ) );
     }
     if( super ) {
	 const QMetaObject * moSuper = mo->superClass();
	 if( moSuper ) {
	     QStrList listSuper = getObjectSlots( QString(moSuper->className()),super);
	     for( unsigned int i = 0 ; i < listSuper.count() ; i++ )
		     list.append(  listSuper.at(i) );
	 }
     }
     return list;
#else
     return obj->metaObject()->slotNames(super);
#endif
}


// probablement source de pb.
QStrList MetaLibrary::getObjectSlots(QString objName, bool super)
{
     QObject* obj = createObject(objName);
     if (obj == NULL)
	  std::cerr << "Required class " << objName.ascii() << " not found." << std::endl;
     QStrList l = getObjectSlots(obj,super);
     //delete obj;
     destroyObject(obj,objName);
     return l;
}
