/*
  name: metaruntime.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include <metalibrary/metaruntime.h>
#include <iostream>
#include <qmetaobject.h>
#include <qstrlist.h>

#include <qapplication.h>
#include <qtimer.h>

using namespace std;

MetaRuntime::MetaRuntime(QObject* parent, const char* name) : QObject(parent, name)
{
     eventDriven = true;
     stopped = false;
     sheetCurrentlyActivated = false;
}


MetaRuntime::~MetaRuntime()
{

}


int MetaRuntime::startRuntime(QString fileName, bool evtDriven)
{
     setEventDriven(evtDriven);
     if (!loadConfigFile(fileName))
	  return 2;

     if (!loadLibraries())
	  return 3;
     if (!instanciateObjects())
	  return 4;
     if (!instanciateSheets())
	  return 5;
     if (!instanciateStateMachine())
	  return 6;


     if (evtDriven)
     {
	  QObject::connect(this,SIGNAL(finished()), qApp, SLOT(closeAllWindows()));
	  QObject::connect(this,SIGNAL(finished()), qApp, SLOT(quit()));

	  cout << "Waiting 1s before launching runtime ..." << endl;
	  QTimer::singleShot( 1000, this, SLOT(run()));
	  return qApp->exec();
     }
     else
     {
	  run();
	  while (!hasFinished())
	  {
	       if (!sheetCurrentlyActivated ) //&& !msm.isTerminalState())
		    nextSheet();
	  }
     }
     return 0;

}


bool MetaRuntime::loadConfigFile(QString filename)
{
     return mcr.openFile(filename);
}

//! \todo split this method in two, first parsing then loading
bool MetaRuntime::loadLibraries(bool parse)
{
    /* This function is called twice : 
        one when the main file is loaded and one when the object are loaded
        this is to circumvent troubles due to additionnal libraries to 
        load 
     */
     if (parse)
     {
         if (!mcr.parseLibrariesSection())
         {
    	  cerr << "[META] Failed to parse libraries section." << endl;
	     return false;
        }
     }

     QStringList libs = mcr.getLibraryNames();
     
     for (QStringList::Iterator it = libs.begin(); it != libs.end(); ++it)
     {
//	  cout << "Loading library " << (*it).ascii() << endl;
	  if (!ml.loadLibrary(*it))
	  {
	       cerr << "[Meta] Library missing : " << (*it).ascii() << endl;
	       return false;
	  }
     }
     return true;
}




bool MetaRuntime::instanciateObject(MetaObject* mobj)
{
//     cout << "Attempting to initialize " << mobj->getID().ascii() << " of class " << mobj->getClassName().ascii() << mobj->isBlock() <<  endl;


     if (mobj->isBlock())
     {
	  MetaBlock* mb = (MetaBlock*)mobj;
	  if (mb->isInstanciated())
	  {
	       cerr << "[Meta] Attempted to re-initialize metablock" << endl;
	       return true;
	  }
	  QStringList lst = mb->getObjects();
	  for (unsigned int i=0; i <lst.count() ; i++)
	  {
	       MetaObject* mo = MetaObject::getObject(lst[i]);
	       if (mo == NULL)
		    return false;
	       if (!instanciateObject(mo))
		    return false;
	  }
	  if (mb->getSheet())
	       activateSheet(mb->getSheet());
	  mb->setInstanciated(true);
	  return true;
     }

     if (mobj->getReference() != NULL)
     {
	  //cerr << "[Meta] Attempted to instanciate an object already instanciated : " << mobj->getID().ascii() << endl;
	  return true;
     }
     QObject* obj = ml.createObject(mobj->getClassName(), this, mobj->getID().ascii());
     if (obj == NULL)
     {
	  cerr << "[META] Object " << mobj->getID().ascii() << " couldn't be instanciated." << endl;
	  return false;
     }
     else
     {
	  mobj->setReference(obj);
#ifdef DEBUG
//	  cout << "##### Dumping slots : " << endl;
//	  QStrList slt = mobj->metaObject()->slotNames(true);
//	  for (unsigned int i = 0; i < slt.count() ; i++)
//	       cout << slt.at(i) << endl;
#endif
     }
     return true;
}


bool MetaRuntime::destroyObject(MetaObject* mobj)
{
     if (mobj->isBlock())
     {
	  MetaBlock* mb = (MetaBlock*)mobj;

	  if (!mb->isInstanciated())
	  {
	       cerr << "[Meta] Attempted to destroy empty metablock" << endl;
	       return true;
	  }

	  if (mb->getSheet())
	       deactivateSheet(mb->getSheet());

	  QStringList lst = mb->getObjects();
	  for (unsigned int i=0; i <lst.count() ; i++)
	  {
	       MetaObject* mo = MetaObject::getObject(lst[i]);
	       if (mo == NULL)
		    return false;
	       if (!destroyObject(mo))
		    return false;
	  }
	  mb->setInstanciated(false);
	  return true;
     }

     if (mobj->getReference() == NULL)
     {
	  cerr << "[Meta] Cannot destroy an object already destroyed : " << mobj->getID().ascii() << endl;
	  return false;
     }
     ml.destroyObject(mobj->getReference(), mobj->getClassName());
     mobj->setReference(NULL);
     return true;
}


bool MetaRuntime::instanciateObjects()
{
     int failed = 0;
     if (!mcr.parseObjectsSection())
     {
	  cerr << "[META] Failed to parse objects section." << endl;
	  return false;
     }
     // just put here code to reload libraries
     if (!loadLibraries(false))
     {
	  return false;

     }



     QDictIterator<MetaObject> it = MetaObject::getIterator();

     for (; it.current(); ++it)
     {
	  if (it.current()->isPersistent())
	  {
	       if (!instanciateObject(it.current()))
		    failed ++;
	  }
     }

     if (failed > 0)
     {
	  cerr << "[META] " << failed << " objects couldn't be instanciated." << endl;
	  return false;
     }
     return true;
}


bool MetaRuntime::instanciateSheets()
{
     if (!mcr.parseSheetsSection())
     {
	  cerr << "[META] Failed to parse sheets section." << endl;
	  return false;
     }
     
     // Put here some additionnal checking ie, if objects used are instanciated.
     return true;
}


bool MetaRuntime::instanciateStateMachine()
{
     if (!mcr.configureStateMachine(&msm))
     {
	  cerr << "[META] Failed to parse state machine section." << endl;
	  return false;
     }
     // Put here some additionnal checking ie, if sheets used are instanciated.
     return true;
}


void MetaRuntime::run()
{
     stopped = false;
     cout << "Launching Runtime, go go go !" << endl;
     cout << "==========================================================" << endl;
     msm.start();
#ifdef DEBUG
     cout << "Current State " << msm.getCurrentState().ascii() << endl;
#endif
     if(msm.isTerminalState())
     {
	  // Some cleanup must be achieved here so that the event loop is saved.
	  cout << "Finishing ..." << endl;
	  msm.clear();
	  MetaSheet::clear();
	  MetaObject::clear();
	  stopped = true;
	  emit finished();
	  return ;
     }
     activateSheet(msm.getCurrentState());
}


void MetaRuntime::recieveToken(QString token)
{
#ifdef DEBUG
     cout << "Recieved a token " << token.ascii() << endl;
#endif
     QString oldstate = msm.getCurrentState();
     if (!msm.changeState(token))
     {
	  cerr << "[META] Unknown transition." << endl;
	  //emit finished();
	  return;
     }


     deactivateSheet(oldstate);
     if(msm.isTerminalState())
     {
#ifdef DEBUG
	  cout << "Reached a terminal state, sending a finished signal." << endl;
#endif
	  // Some cleanup must be achieved here so that the event loop is saved.
	  cout << "Finishing ..." << endl;
	  msm.clear();
	  MetaSheet::clear();
	  MetaObject::clear();
	  cout << "Emitting finished" << endl;
	  stopped = true;
	  emit finished();
	  return ;
     }
#ifdef DEBUG
     cout << "Current state is :" << msm.getCurrentState().ascii() << endl;
#endif

     if (eventDriven)
	  QApplication::postEvent(this, new QEvent(QEvent::User));
     //activateSheet(msm.getCurrentState());
}

bool MetaRuntime::event(QEvent* e)
{
     if (e->type() != QEvent::User)
	  return false;

     activateSheet(msm.getCurrentState());
     return true;
}


void MetaRuntime::activateSheet(MetaSheet* ms)
{
     int i,pc;

#ifdef DEBUG
     cout << "Activating sheet " << ms->getSheetID().ascii() << endl;
#endif 

     pc = ms->getPreConnectionsNumber();
     for(i=0; i < pc; i++)
     {
	  MetaInit mi = ms->getPreConnection(i);
	  initialize(mi);//mi.getObject(), mi.getSlot(), mi.getType(), mi.getValue());
     }

     pc = ms->getConnectionsNumber();
     for (i=0; i <pc; i++)
	  connect(ms->getConnection(i));

     // connecting token sender
//     cout << "coucou" << endl;

     QString a  = ms->getTokenSender();
     if ( a != QString::null)
     {
	  QString b = "sendToken(QString)";
	  MetaConfigReader::getSignalCouple(a,b);
	  b = "2" + b ;

	  QObject* obj = MetaObject::getObject(a)->getReference();
	  QObject::connect(obj,b.ascii(),this,SLOT(recieveToken(QString)));
     }

     // doit tre plac avant le batch des post-connections sinon la feuille ne change jamais
     sheetCurrentlyActivated = true;
	  
     pc = ms->getPostConnectionsNumber();
     for(i=0; i < pc; i++)
     {
	  MetaInit mi = ms->getPostConnection(i);
	  initialize(mi);//mi.getObject(), mi.getSlot(), mi.getType(), mi.getValue());
     }

}


void MetaRuntime::deactivateSheet(MetaSheet* ms)
{
     int i,pc;

#ifdef DEBUG
     cout << "Deactivating sheet " << ms->getSheetID().ascii() << endl;
#endif 


     pc = ms->getConnectionsNumber();
     for (i=0; i <pc; i++)
	  disconnect(ms->getConnection(i));

     QString a  = ms->getTokenSender();
     QString b = "sendToken(QString)";
     MetaConfigReader::getSignalCouple(a,b);
     b = "2" + b ;

     QObject* obj = MetaObject::getObject(a)->getReference();
     QObject::disconnect(obj,b.ascii(),this,SLOT(recieveToken(QString)));
     sheetCurrentlyActivated = false;

}



void MetaRuntime::initialize(MetaInit mi)//QString objectID, QString slot, int type, QString value)
{
#ifdef DEBUG
     cout << "Initializing " << mi.getObject().ascii() << " on slot " 
	  << mi.getSlot().ascii() << " with value " << mi.getValue().ascii() << endl;
#endif
     MetaObject* mobj = mi.getObjectP();//MetaObject::getObject(objectID);
     
     QString objectID = mi.getObject();
     QString slot = mi.getRealSlot() ; // ajout
     QString value = mi.getValue(); // ajout
     int type = mi.getType();

     
     // Cette portion de code gre le cycle de vie des objets
     if (type == MetaInit::Object)
     {
	  if (objectID != "__this__")
	  {
	       if (slot == "instanciate")
	       {
		    if (mobj->isPersistent())
		    {
			 cerr << "[Meta] Instanciating persistent object is forbidden  : " << objectID.ascii() <<endl; 
			 return;
		    }
		    if (!instanciateObject(mobj))
			 QApplication::exit(0);

	       }

	       if (slot == "destroy" )
	       {
		    if (mobj->isPersistent())
		    {
			 cerr << "[Meta] Destroying persistent object is forbidden  : " << objectID.ascii() <<endl; 
			 return;
		    }
		    if (!destroyObject(mobj))
			 return;
	       }
	  }
     } 


     QObject* obj = (mi.getObject() == "__this__" )?this:mobj->getReference();
     //QString s1 = "1" + slot; // ceci est un hack de Qt 3.x 
     QString s1 = slot;
     void* valueData = mi.getValueData();

     switch(type)
     {
     case MetaInit::Void:
     {
	  QObject::connect(this,SIGNAL(init()), obj,s1.ascii());
	  emit init();
	  QObject::disconnect(this,SIGNAL(init()),obj, s1.ascii());
	  break;
     }
     case MetaInit::Bool:
     {
	  QObject::connect(this,SIGNAL(init(bool)), obj,s1.ascii());
	  //if (value.lower()== "true")
	  //    emit init(true);
	  //if (value.lower()== "false")
	  //     emit init(false);
	  //if (ok)
	  //     if (val)
	  //	    emit init(true);
	  //     else
	  //    emit init(false);
	  emit init(*(bool*)mi.getValueData());
	  QObject::disconnect(this,SIGNAL(init(bool)), obj,s1.ascii());
	  break;
     }
     case MetaInit::Integer:
     {
	  QObject::connect(this,SIGNAL(init(int)),obj,s1.ascii());
	  emit init(*(int*)mi.getValueData());
	  QObject::disconnect(this,SIGNAL(init(int)),obj,s1.ascii());
	  break;
     }
     case MetaInit::Double:
     {
	  QObject::connect(this,SIGNAL(init(double)),obj,s1.ascii());
	  emit init(*(double*)mi.getValueData());
	  QObject::disconnect(this,SIGNAL(init(double)),obj,s1.ascii());
	  break;
     }
     case MetaInit::Float:
     {
	  QObject::connect(this,SIGNAL(init(float)),obj,s1.ascii());
	  emit init(*(float*)mi.getValueData());
	  QObject::disconnect(this,SIGNAL(init(float)),obj,s1.ascii());
	  break;
     }
     case MetaInit::String:case MetaInit::Path:
     {
	  QObject::connect(this,SIGNAL(init(QString)),obj,s1.ascii());
	  emit init(value);
	  QObject::disconnect(this,SIGNAL(init(QString)),obj,s1.ascii());
	  break;
     }
     case MetaInit::Object:
     {
          QObject::connect(this,SIGNAL(init(QObject*)),obj,s1.ascii());
	  // some objects might not be in cache at this point 
	  // so we should check it first.
	  if (mi.getValueData() == NULL)
	       mi.convertValue();
          emit init((QObject*)mi.getValueData());
          QObject::disconnect(this,SIGNAL(init(QObject*)),obj,s1.ascii());
     }
     }
}


void MetaRuntime::connect(MetaWire mw)
{
#ifdef DEBUG 
     cout << "Connection from" << mw.getSender().ascii() <<  ",signal " << mw.getSignal().ascii() 
	  << " to " << mw.getReciever().ascii() << ", slot " << mw.getSlot().ascii() <<  endl;
#endif 
     QString slt = mw.getRealSlot();
     QString sgn = mw.getRealSignal(); 

     QObject* objA = (mw.getSender() == "__this__")?this:mw.getSenderObject()->getReference(); 
     QObject* objB = (mw.getReciever() == "__this__")?this:mw.getRecieverObject()->getReference(); 
     QObject::connect(objA,sgn.ascii(), objB, slt.ascii());
}


void MetaRuntime::disconnect(MetaWire mw)
{
     QString slt = mw.getRealSlot(); 
     QString sgn = mw.getRealSignal(); 
     QObject* objA = (mw.getSender() == "__this__")?this:mw.getSenderObject()->getReference(); 
     QObject* objB = (mw.getReciever() == "__this__")?this:mw.getRecieverObject()->getReference(); 
     QObject::disconnect(objA, sgn.ascii(), objB, slt.ascii());
}



void MetaRuntime::dump()
{
     QStringList libs = mcr.getLibraryNames();     
     cout << "##### Dumping libraries :" << endl; 
     for (QStringList::Iterator it = libs.begin(); it != libs.end(); ++it)
	  cout << (*it).ascii() << endl;

     cout << "##### Dumping objects :" << endl;

     QDictIterator<MetaObject> it = MetaObject::getIterator();
     for (; it.current(); ++it)
	  cout << *(it.current()) << endl;
     
     cout << "##### Dumping sheets :" << endl;

     QDictIterator<MetaSheet> it2 = MetaSheet::getIterator();
     for (; it2.current(); ++it2)
	  cout << *(it2.current()) << endl;

     cout << "##### Dumping state machine :" << endl;
     cout << msm << endl;
     


     cout << "End of Dump #####" << endl;
}
