/*
  name: metabuilder.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include <metalibrary/metaconfigreader.h>
#include <iostream>
#include <iomanip>
#include <qstring.h>
#include <qmap.h>
#include <qstringlist.h>
#include <qstrlist.h>
#include <qapplication.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qfile.h>
#include <qregexp.h>
#include <qtextstream.h>

#include <qprocess.h>
#include <stdlib.h>
#include <unistd.h>


using namespace std;


QMap<QString,QString> libraryDirBounds;

/*! \brief Strips the library name from a given path
 *
 */
QString getLibraryName(QString libpath)
{
     QString lname = libpath.right(libpath.length() - libpath.findRev(QDir::separator())-1 );
     lname = lname.left(lname.findRev('.'));
     if (lname.startsWith("lib"))
     {
	  lname = lname.right(lname.length()-3);
     }

     return lname;     
}


void exploreProjectFile(QString profile)
{
     // First : strip the name of the pro file*
     QFileInfo fi(profile) ;
     QString name = fi.baseName();
     QString path = fi.dirPath();

     // Second open the file and detect lines 
     // TARGET = something 
     QString target;
     
     QFile filepro(profile);
     filepro.open(IO_ReadOnly);
     QTextStream ts(&filepro);
     QRegExp linematcher("^\\s*TARGET\\s*=\\s*(\\w+)\\W*$");


     QString line;
     do 
     {
	  line = ts.readLine();
	  if (linematcher.exactMatch(line))
	  {
	       target = linematcher.cap(1);
	  }
     }
     while ( !line.isNull());

     // select what to put inside mapper 
     if (target.isEmpty())
     {
	  std::cout << std::setw(15) << name.ascii() << " => " << path.ascii() << std::endl;
	  libraryDirBounds[name] = path;
     }
     else
     {
	  std::cout << std::setw(15) << target.ascii() << " => " << path.ascii() << std::endl;
	  libraryDirBounds[target] = path; 
     }
}



void buildBounds(QString path)
{
     unsigned int i;
     QDir d(path);

     // Make sure path is '/' terminated
     if (path.at(path.length()-1) != QDir::separator())
	  path += QDir::separator();

     // First pass : scan all .pro files 
     QStringList profiles = d.entryList("*.pro");
     for (i=0; i < profiles.count() ; i++)
	  exploreProjectFile(path + profiles[i]);

     
     // Second pass : scan all directories
     QStringList dirs = d.entryList(QDir::Dirs);
     for (i=0; i < dirs.count() ; i++)
     {
	  // filter out directories with leading '.'
	  if (dirs[i].at(0) != '.' )
	       buildBounds(path + dirs[i]);
     }
}


bool buildLibrary(QString path)
{
     cout << "    QMaking";
     QProcess pr(QString("qmake"));
     pr.setWorkingDirectory(QDir(path));
 
     pr.start();
     while(pr.isRunning())
     {
	  cout << "." ;
	  cout.flush();
	  sleep(1);
     }
     cout << endl;

     if ( pr.normalExit() && (pr.exitStatus() == 0))
     {

	  pr.clearArguments();
	  pr.addArgument(QString("make"));
	  cout << "    Making" ;
	  pr.start();
	  while(pr.isRunning())
	  {
	       cout << "." ;
	       cout.flush();
	       sleep(1);
	  }
	  cout << endl;
	  if (pr.normalExit() && (pr.exitStatus() == 0))
	  {
	       return true;
	  }
	  else
	       return false;
     }
     else
	  return false;
}


int buildApplication(QStringList libs, QString path)
{
     int success = 0;

     std::cout << "Looking for bindings between libs and directories" << std::endl;
     buildBounds(path);

     for (int i=0; i < libs.count() ; i++)
     {
	  QString libname = getLibraryName(libs[i]);
	  std::cout << "Trying to build library " << libname.ascii() << std::endl;
	  QString dir = libraryDirBounds[libname];
	  if (!dir.isEmpty())
	  {
	       if (buildLibrary(dir))
	       {
		    std::cout << std::setw(10) << " " << " Success !" << std::endl;
		    success++;
	       }
	       else
		    std::cout << std::setw(10) << " " <<  "Failed !" << std::endl;
		    
	       
	  } 
	  else
	       std::cout << std::setw(10) << " " << "Failed !"  << std::endl;
     }
     return success;
}






void usage(char* name)
{
	  cerr << "Usage : " << name << " application_file.xml [component_sources_path]" << endl;
	  cerr << "Where : application_file.xml is an application file." << endl;
	  cerr << "        [component_sources_path] is the optionnal source path." << endl;
	  cerr << "This is v2 " << endl;
}

int main(int argc, char* argv[])
{
     QString path;
     QApplication app(argc,argv);
     if ((argc == 1) || (argc > 3))
     {
	  usage(argv[0]);
	  return 1;
     }


     char* buildpath = getenv("METABUILDPATH");


     if (buildpath == NULL)
	  cerr << "No environment variable for build path found ..." << endl;
     else
	  path = QString(buildpath);


     if ((buildpath == NULL) && (argc < 3))
     {
	  usage(argv[0]);
	  return 1;
     }



     if (argc == 3)
	  path = QString(argv[2]);

     MetaConfigReader mcr;
     if (mcr.openFile(argv[1]))
     {
	  if (mcr.parseLibrariesSection())
	  {
	       int res = buildApplication(mcr.getLibraryNames(), path);
	       if (res != mcr.getLibraryNames().count())
		    cout << "Failed to build some libraries required by your application." << endl;
	       else
		    cout << "All libraries successfully built." << endl;
		   
	  }
	  else
	  {
	       cerr << "Failed to parse libraries section" <<endl;
	       return 1;
	  }
     }
     else
     {
	  cerr << "Failed to load file " << argv[1] << endl;
	  return 1;
     }

     return 0;
}
