/*
  name: objectcanvas.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __OBJECTCANVAS_H__
#define __OBJECTCANVAS_H__

/*! \file transitioncanvas.h
 * \author Yoann Petit
 * \date 6 Juillet 2005
 * 
 * File describes ObjectCanvas
 */

#include "rtti.h"

#include <qcanvas.h>
#include <qdict.h>
#include <qptrlist.h>
#include <metalibrary/metabase.h>
#include "wirecanvas.h"
#include <metalibrary/metalibrary.h>
#include "initcanvas.h"
#include "iocanvas.h"

/*! \brief Object CanvasItem description
 * 
 * This is the graphical counterpart of Object included in a sheet
 */
class ObjectCanvas : public QCanvasRectangle
{
public:
     
     /*! \brief ObjectCanvas constructor.
      *  \param *mo MetaObject of the object.
      *  \param x ObjectCanvas position along the X-axis on the sheet.
      *  \param y ObjectCanvas position along the Y-axis on the sheet.
      *  \param width Width of the object.
      *  \param height Height of the object.
      *  \param *canvas Canvas where is the ObjectCanvas.
      *  \param *ml MetaLibrary of the application.
      */
     ObjectCanvas(MetaObject* mo, int x, int y, int width, int height, QCanvas* canvas);
     ~ObjectCanvas();

     /*! \brief Returns the RTTI of the ObjectCanvas.
      *  \return RTTI of the ObjectCanvas.
      */
     int rtti() const  { return RTTI_OBJECTCANVAS;} 

     /*! \brief Overload function of moveBy.
      *  Moving of all elements of the ObjectCanvas.
      *  \param dx ObjectCanvas's moving along the X-axis.
      *  \param dy ObjectCanvas's moving along the Y-axis.
      */
     void moveBy(double dx, double dy);

     /*! \brief Returns the width of the objectCanvas.
      *  \return Width of the objectCanvas.
      */
     int getWidth(){ return maxWidth; }

     /*! \brief Returns the height of the objectCanvas.
      *  \return Height of the objectCanvas.
      */
     int getHeight(){ return maxHeight; }

     /*! \brief Returns the ID of the objectCanvas.
      *  \return ID of the objectCanvas.
      */
     QString getObjectID(){return myObject->getID();}
     
     /*! \brief Returns the MetaObject of the objectCanvas.
      *  \return Pointer to the MetaObject of the objectCanvas.
      */
     MetaObject* getMetaObject() { return myObject; }

     /*! \brief Adds a NodeItem to an object.
      *  \param nodeID ID of the NodeItem in the object.
      *  \param slot Sets the type of the node, true for a slot, false for a signal.
      */
     void addNode(QString nodeID, bool slot);

     /*! \brief Returns a NodeItem with the slot "id".
      *  \param id Name of the NodeItem' slot.
      *  \return NodeItem with the slot "id".
      */
     NodeItem* getNodeSlot(QString id) { return dNodeSlot[id]; }

     /*! \brief Returns a NodeItem with the signal "id".
      *  \param id Name of the NodeItem' signal.
      *  \return NodeItem with the signal "id".
      */
     NodeItem* getNodeSignal(QString id) { return dNodeSignal[id]; }

     /*! \brief Register a pre or post-connection in a dictionnary of InitCavans.
      *  \param *ic Pointer to the InitCanvas of the pre or post-connection.
      */
     void registerInit(InitCanvas* ic);

     /*! \brief Adds InitCanvas to the MetaSheet.
      *  \param *ms Pointer to the MetaSheet.
      */
     void setInit(MetaSheet *ms);


     void setObjectPersistence(bool sw);


     /*! \brief Removes a connection.
      *  \param *wc Pointer to the WireCanvas.
      */
     void delInit(QString s) {dInit.remove(s);}

     InitCanvas* getFirstInit(QString & key) 
	  { 
	       QDictIterator<InitCanvas> dic(dInit);
	       key = dic.currentKey();
	       return dic.current();
	  }
     //WireCanvas* getFirstWire() { return QDictIterator<WireCanvas>(dInit).toFirst(); }


     /*! \brief Returns a pre or post-connection conects on the slot "id".
      *  \param id Name of the InitCanvas' slot.
      *  \return InitCanvas connects on the slot "id".
      */
     InitCanvas* getInit(QString id) { return dInit[id]; }

     /*! \brief Returns an iterator to the dictionnary of InitCanvas.
      *  \return Iterator to the dictionnary of InitCanvas.
      */
     QDictIterator<InitCanvas> getInitIterator() { return QDictIterator<InitCanvas>(dInit); }
 
     /*! \brief Register a connection in a dictionnary of WireCanvas.
      *  \param *wc Pointer to the WireCanvas of the connection.
      */
     void registerWire(WireCanvas* wc);
     /*! \brief Adds WireCanvas to the MetaSheet.
      *  \param *ms Pointer to the MetaSheet.
      */
     void setWire(MetaSheet *ms);

     /*! \brief Removes a connection from the dictionnary of WireCanvas.
      *  \param slt Slot of the connection.
      */
     void delWireSlot(QString slt) { dWireSlot.remove(slt); }

     /*! \brief Removes a connection from the dictionnary of WireCanvas.
      *  \param slt Slot of the connection.
      */
     void delWireSignal(QString sgn) { dWireSignal.remove(sgn); }

     /*! \brief Removes a connection.
      *  \param *wc Pointer to the WireCanvas.
      */
     void delWire(WireCanvas* wc);
     
     void registerIO(IOCanvas* io) { dIO.insert(io->getMetaIO()->getPublicName(),io); }
     bool hasIO(QString s) { return ((s.isEmpty())?false:(dIO[s] != NULL)); }

     void delIO(QString s) {dIO.remove(s); }


     /*! \brief Clears all connections of the object.
      */
     void clear() { dWireSlot.clear(); dWireSignal.clear(); }
 
     /*! \brief Returns a list of pointer to the WireCanvas with a signal "id".
      *  \param id Name of a signal.
      *  \return list of pointer to the WireCanvas with a signal "id".
      */
     QPtrList<WireCanvas> getWireSignal(QString id);

     /*! \brief Returns a list of pointer to the WireCanvas with a slot "id".
      *  \param id Name of a slot.
      *  \return list of pointer to the WireCanvas with a slot "id".
      */
     QPtrList<WireCanvas> getWireSlot(QString id);

     /*! \brief Returns an iterator to the dictionnary of WireCanvas connected to a signal.
      *  \return Iterator to the dictionnary of WireCanvas connected to a signal.
      */
     QDictIterator<WireCanvas> getWireIterator() { return QDictIterator<WireCanvas>(dWireSignal); }

     /*! \brief Returns an iterator to the dictionnary of WireCanvas connected to a slot.
      *  \return Iterator to the dictionnary of WireCanvas connected to a slot.
      */
     QDictIterator<WireCanvas> getWireIteratorSlot() { return QDictIterator<WireCanvas>(dWireSlot); }



     QDictIterator<IOCanvas> getIOIterator() { return QDictIterator<IOCanvas>(dIO); }


private:

     MetaObject* myObject;
     QPtrList<QCanvasItem> myItem;
     QString label;
     int posSlt;
     int posSgn;
     QCanvasText* titlelabel;
     QCanvasText* underlabel;
     QDict<WireCanvas> dWireSignal;
     QDict<WireCanvas> dWireSlot;
     QDict<NodeItem> dNodeSignal;
     QDict<NodeItem> dNodeSlot;
     QDict<IOCanvas> dIO;
     QDict<InitCanvas> dInit;
     int maxHeight;
     int maxWidth;
     int maxSltLength;
     int maxSgnLength;
};

#endif //__OBJECTCANVAS_H__
