/*
  name: sheetview.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef SHEETVIEW_H
#define SHEETVIEW_H

/*! \file sheetview.h
 * \author Yoann Petit
 * \date 5 Juillet 2005
 * 
 * File describing SheetView class
 */

#include <qcanvas.h>
#include <qtoolbutton.h>
#include <qdict.h>
#include "objectcanvas.h"
#include <metalibrary/metabase.h>
#include "wirecanvas.h"
#include <metalibrary/metalibrary.h>
#include <qpoint.h>
//#include <qqueue.h>
#include <qevent.h>
#include <vector>
#include "sheettooltip.h"
#include "iconcomp.h"
#include "iconblck.h"

#include <iostream>


typedef std::vector<InitCanvas*> InitVector;
typedef std::vector<WireCanvas*> WireVector;



/*! \brief SheetView description
 * 
 * This is the graphical counterpart of sheet included
 */
class SheetView : public QCanvasView
{
    Q_OBJECT
public:
    
    /*! \brief SheetView constructor.
     *  \param *parent Parent of the StateMachineView.
     *  \param *name Name of the StateMachineView.
     *  \param tokenSenderID ID of the object token sender.
     */
    SheetView( QWidget *parent, const char *name, QString tokenSenderID = 0 );
    ~SheetView();

    /*! \brief Returns The ID of the token sender object of a sheet.
     *  \return ID of the Token sender object.
     */
    QString getTokenSender() { return tokenSenderID; }

    /*! \brief Sets the token sender object of a sheet.
     *  \param ID ID of the token sender object
     */
    void setTokenSender(QString ID);

    /*! \brief Adds InitCanvas to the MetaSheet.
     *  \param *ms Pointer to the MetaSheet.
     */
    void setInit(MetaSheet * ms);

    /*! \brief Adds WireCanvas to the MetaSheet.
     *  \param *ms Pointer to the MetaSheet.
     */
    void setWire(MetaSheet * ms);

    void setAsMacroblock(bool b) 
	 { 
	      macroblock=b; 
	      if (b) 
	      {
		   canvas()->setBackgroundColor(QColor(240,240,255)); 
		   cornerImage->setIconSet(QPixmap((const char**)iconblck));
	      }
	      else 
	      {
		   canvas()->setBackgroundColor(Qt::white);
		   cornerImage->setIconSet(QPixmap((const char**)iconcomp));
	      }
	 }
    bool isMacroblock() { return macroblock;}


    /*! \brief Resets all pointers.
     */
    void resetPointers();

    /*! \brief Returns a pointer to the ObjectCanvas id.
     *  \param id ID of a ObjectCanvas.
     *  \return ObjectCanvas if exists, otherwise 0 if the object does not exist in the dictionary.
     */
    ObjectCanvas* getObjectCanvas(QString id) {return dObjectCanvas[id];}

    /*! \brief Remove a ObjectCanvas in the dictionnary of Objects.
     *  \param ID ID of an Object.
     */
    void delObjectCanvas(QString ID); // { dObjectCanvas.remove(ID); }

    bool hasObject(QString s) { bool b=false; hasObjectInSheet(s,&b); return b;}

    void delInit(InitCanvas* ic);


    InitVector getInitVector(bool post) { return (post)?vPostInit:vPreInit; }

    void switchInit(InitCanvas* ic);

    void delWire(WireCanvas* wc);
    
    WireVector getWireVector() { return vWire; }

    void getTip(QPoint p, QRect& r, QString& s);
    
    /*! \brief Returns an iterator to the dictionnary of ObjectCanvas.
     *  \return Iterator to the dictionnary of ObjectCanvas.
     */
    QDictIterator<ObjectCanvas> getObjectIterator() { return QDictIterator<ObjectCanvas>(dObjectCanvas); }
    
    /*! \brief Returns a vector of InitCanvas.
     *  \return A vector of InitCanvas.
     */
    std::vector<InitCanvas*> getPostInitVector() { return vPostInit; }


protected:

    /*! \brief When the user press a button of the mouse.
     */
    void contentsMousePressEvent( QMouseEvent* );

    /*! \brief When the user moves the mouse. 
     */
    void contentsMouseMoveEvent( QMouseEvent* );

    /*! \brief When the user releases the button of the mouse
     */
    void contentsMouseReleaseEvent( QMouseEvent* );

    void contentsMouseDoubleClickEvent( QMouseEvent* e);

    
    /*! \brief Is called when the user select an ObjectCanvas on a Sheet.
     *  \param *ci Pointer to the objectCanvas.
     *  \param pe Point of start position of the objectCanvas.
     */
    void selectObjectCanvas(QCanvasItem *ci, QPoint pe);

    /*! \brief Is called when the user select an NodeItem on a Sheet.
     *  \param *ci Pointer to the NodeItem.
     */
    void selectNodeItem(QCanvasItem *ci);

    /*! \brief Is called when the user select an WireCanvas on a Sheet.
     *  \param *ci Pointer to the WireCanvas.
     */
    void selectWireCanvas(QCanvasItem *ci);

    /*! \brief Is called when the user select an WireGrip on a Sheet.
     *  \param *ci Pointer to the WireGrip.
     *  \param pe Point of start position of the WireGrip.
     */
    void selectWireGrip(QCanvasItem *ci, QPoint pe);


signals:

    void changeSheet( QString );


    //void sendWire(MetaWire*);
    //void sendInit(MetaInit*);
    //void getNbrObjInSheet( int* );
    
public slots:
     
    /*! \brief Adds an object on a Sheet.
     *  \param *obj Pointer to the MetaObject of the Object.
     */
    void addObject(MetaObject* obj );

    /*! \brief Adds a connection between two objects on a Sheet.
     *  \param *wire Pointer to the MetaWire of the connection.
     */
    void addWire(MetaWire* wire);

    /*! \brief Adds a pre-connection or post-connection for an objects on a Sheet.
     *  \param *init Pointer to the MetaInit of the connection.
     *  \param postc Sets true if is a post-connection, otherwise false.
     */
    void addInit(MetaInit* init, bool postc);


    void addIO(MetaIO* io, bool isSlot);

    /*! \brief Allows to know if an object is on a sheet.
     *  \param ID ID of the object.
     *  \param *b Pointer to a boolean.
     *  A boolean allows to return the answer.
     */
    void hasObjectInSheet(QString ID, bool *b);

    /*! \brief Allows to know how many objects is on a sheet.
     *  \param ID ID of the object.
     *  \param *b Pointer to a interger.
     *  A integer allows to return the answer.
     */
    void giveNbrObjInSheet( int *nbr );

    /*! \brief Sets the vector of InitCanvas with a new vector.
     *  \param vIC The new vector.
     *  \param post Define the type of init, true for a post-connection, false for a pre-connection.
     */
    void setInitVector( InitVector vIC, bool post );

    /*! \brief Returns the vector of InitCanvas.
     *  \param *vIC Pointer to a empty vector.
     *  \param post Define the type of init, true for a post-connection, false for a pre-connection.
     *  *vIc allows to return the vector.
     */
    //void giveInitVector(vector<InitCanvas*> *vIC, bool post);


    /*! \brief Sets the vector of WireCanvas with a new vector.
     *  \param vWC The new vector.
     */
    void setWireVector( WireVector vWC);

    /*! \brief Returns the vector of WireCanvas.
     *  \param *vWC Pointer to a empty vector.
     *  *vIc allows to return the vector.
     */
//    void giveWireVector(vector<WireCanvas*> *vWC);
    
private:
    void contextMenuEvent ( QContextMenuEvent * );
    
    //double depth;
    QDict<ObjectCanvas> dObjectCanvas;
    /*vector<WireCanvas*>*/ WireVector vWire;
    /*vector<InitCanvas*>*/ InitVector vPostInit;
    /*vector<InitCanvas*>*/ InitVector vPreInit;
    
    QCanvas *myCanvas;
    QToolButton *cornerImage;
    QString tokenSenderID;
    SheetTooltip* stt;
    QCanvasItem *Moving;
    NodeItem *Node;
    GripWire *Grip;
    WireCanvas *Wire;
    ObjectCanvas *Object;
    QPoint MovingStart;
    //QQueue<QPoint> MousePosQueue;
    double zSave;
    QPtrList<WireCanvas> plWire;    
    int order;
    bool macroblock;
};

#endif  // SHEETVIEW_H
