/*
  name: metaconfigwriter.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __METACONFIGWRITER_H__
#define __METACONFIGWRITER_H__

#include <qdom.h>
#include <qstringlist.h>

#include <metalibrary/metabase.h>
#include <metalibrary/metalibrary.h>

/*!
 * \todo Should test if it works without black magic now.
 */
//#undef DLL_POINT
//#include <metalibrary/metawin.h>

/*! \file metaconfigwriter.h
 * \author Jean-Yves Didier
 * \date 07 Juin 2005
 *
 * File gathering one class : MetaConfigWriter which role is to write down the content 
 * of an XML description of an application.
 * This class is, for instance, only intended to write the XML description of applications.
 * You can have a look to the file metaconfigreader.h which the reading counter-part. 
 */

/*! \brief Class used to write down the content of the xml formats designed for this runtime.
 * 
 * The end-user must first call the methods related to the functions he wants to write down.
 * 
 * This class allows to write the XML description of applications only.
 * It consists in a set of method to build the XML tree of an application.
 * Such XML tree is build using the QDom* classes of Qt library.
 */

class DLL_POINT MetaConfigWriter 
{
public:
     /*! \brief MetaConfigWriter constructor
      * This constructor can take a namespace as a parameter to handle the writing of macro-block description.
      * \param ns a namespace (default: empty) which is also a filename.
      */
     MetaConfigWriter(QString ns=QString::null);
     ~MetaConfigWriter();

     /*! \brief Writes down the application file.
      *
      * This is logically the last method to be called when writing down the application.
      * Note that it is not yet fully compliant with the last runtime specification.
      * It doesn't work on macro-blocks nor defines
      * \param filename The filename in which we want to write down the XML description of the application.
      * \return <tt>true</tt> if succeeded, <tt>false</tt> otherwise
      */
     bool writeFile(QString filename);

     void reset(); //!< empty method

     /*! \brief Adds the libraries to the XML tree.
      *
      * \param ml a pointer to the MetaLibrary object that stores the libraries
      */
     void addLibraries(MetaLibrary* ml);
     void addObjects(); //!< Adds the objects to the XML tree.
     void addSheets(); //!< Adds the sheets to the XML tree.
     void addDefines(); //!< Adds the defines to the XML tree.

     
     //void addSignal(); //seems deprecated

     /*! \brief Sets the terminal state for the state machine.
      *
      * \param s Name of the state.
      */
     void setTerminalState(QString s);
     /*! \brief Sets the initial state for the state machine.
      *
      * \param s Name of the state.
      */
     void setInitialState(QString s);

     /*! \brief Adds a transition to the state machine.
      * 
      * \param a Name of the departure state.
      * \param b Name of the ending state.
      * \param token Token used to trigger transition.
      */
     void addTransition(QString a, QString b, QString token);
     

     /*! \brief Adds a meta-method (signal or slot) in case of writing down a macro-block.
      * \param mio a pointer to a MetaIO instance
      * \param slot set it to <tt>true</tt> if teh considered method is a slot, <tt>false</tt> otherwise.
      */
     void addMethod(MetaIO* mio, bool slot);


private:

     void addSheet(MetaSheet* ms);
     void addDefine(QString id, QString type, QString value);
     void addObject(QString id, QString cn, bool persistent,bool b);

     QString computeRelativePath(QString fp, QString lp);

     void processLibrariesPath(QString filepath);
     void processObjectsPath(QString filepath);
     void setTypeAttribute(QDomElement e, int type);

     // Master markup for applications
     QDomDocument application;
     
     // Common markups for applications and macro-blocks
     QDomElement eLibraries;
     QDomElement eObjects;
     QDomElement eDefines;

     // Specific markups for applications
     QDomElement eSheets;
     QDomElement eStateMachine;

     // Specific markups for macro-blocks
     QDomElement eSheet;
     QDomElement eSignals;
     QDomElement eSlots;

     bool isBlock;
     QString nameSpace;

     QStringList blockList;
     MetaLibrary* libraries;
};


#endif //__METACONFIGWRITER_H__
