/*
  name: metabase.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include <metalibrary/metabase.h>
#include <iostream>

/************************************************
 * MetaObject class
 ***********************************************/
MetaObjectList MetaObject::mol;


MetaObject::MetaObject(QString cn, QString i, bool p)
{
     className = cn;
     objectID = i;
     persistent = p;
     processed = false;
     reference = NULL;
     registerObject();
}


MetaObject::~MetaObject()
{
     //if (reference != NULL)
     //  delete reference;

     mol.remove(objectID);
}


void MetaObject::registerObject()
{
     mol.insert(objectID,this);
}


void MetaObject::clear()
{
     QDictIterator<MetaObject> it = getIterator();
#ifndef WIN32
     delete it.current();
#endif 
     for( ; it.current() ; ++it)
     {
	  delete it.current();
     }
}


std::ostream& operator<<(std::ostream& os, const MetaObject& mo)
{
     os << "Object id:" << mo.objectID.ascii() << ", class:" << mo.className.ascii()  ;
     return os; 
}

#ifdef WIN32 // for msvc compatibility
inline QString MetaObject::getClassName()
{	
	return className; 
}

inline MetaObject* MetaObject::getObject(QString s)
{
	return mol[s]; 
}

inline QDictIterator<MetaObject> MetaObject::getIterator()
{
	return QDictIterator<MetaObject>(mol); 
}
#endif //WIN32
     





/*************************************************
 * MetaIO class
 ************************************************/

MetaIO::MetaIO(QString pn, QString obj, QString ion)
{
     publicName = pn;
     objectName = obj;
     object = MetaObject::getObject(objectName);
     IOObjectName = ion;
}





/************************************************
 * MetaDefine class
 ***********************************************/
MetaDefineList MetaDefine::mdl;

MetaDefine::MetaDefine(QString i, QString t, QString v)
{
     id = i;
     type = t;
     value = v;
     registerDefine();
     mdl.setAutoDelete(true);
}
     
MetaDefine::~MetaDefine()
{
     mdl.remove(id);
}

void MetaDefine::registerDefine()
{
     mdl.insert(id,this);
}

// inline defined for msvc.
#ifdef WIN32
inline QString MetaDefine::getType()
{ 
	return type; 
}

inline QDictIterator<MetaDefine> MetaDefine::getIterator()
{ 
	return QDictIterator<MetaDefine>(mdl);
}
#endif // WIN32

void MetaDefine::clear()
{
     mdl.clear();
}

/*************************************************
 * MetaInit class
 ************************************************/
MetaInit::MetaInit(QString obj, QString sig, QString t, QString v,bool editMode)
{
     objectName = obj;
     if (obj == "this")
	  object = NULL;
     else
	  object = MetaObject::getObject(objectName);
    
     if ((t == "define") && (!editMode) )
     {
	  MetaDefine* def = MetaDefine::getDefine(v);
	  if (def != NULL)
	  {
	       type = convertStringType(def->getType());
	       value = def->getValue();
	  }
     }	
     else
     {
	  type = convertStringType(t);
	  value = v;
     }
     slotName = sig;
     if ((sig == "instanciate") || (sig == "destroy") )
	  realSlotName = sig; 
     else
	  realSlotName = SLOT_PREFIX + sig;

     valueData = NULL;
     convertValue();

}

MetaInit::~MetaInit()
{
     // There is a small memory leak because we cannot deallocate valueData.
}


int MetaInit::convertStringType(QString s)
{
     if (s=="int") return Integer;
     if (s=="double") return Double;
     if (s=="float") return Float;
     if (s=="string") return String;
     if (s=="bool") return Bool;
     if (s=="void") return Void;
     if (s=="object") return Object;
     if (s=="define") return Define;
     if (s=="path") return Path;
     return Undefined;
}


void MetaInit::convertValue()
{
     bool ok;
     int val = value.toInt(&ok);

     switch(type)
     {
     case MetaInit::Bool:
	  if (value.lower()== "true")
	       valueData = new bool(true);
	  if (value.lower()== "false")
	       valueData = new bool(false);
	  if (ok)
	       if (val)
		    valueData = new bool(true);
	       else
		    valueData = new bool(false);
	  break;
     case MetaInit::Integer:
	  valueData = new int(value.toInt());
	  break;
     case MetaInit::Double:
	  valueData = new double(value.toDouble());
	  break;
     case MetaInit::Float:
	  valueData = new float(value.toFloat());
	  break;
     case MetaInit::Object:
     {
          valueData = MetaObject::getObject(value)->getReference();
	  break;
     }
     }
}

bool MetaInit::operator<(const MetaInit& mi)
{
	return ((objectName + value) < (mi.objectName + mi.value));
}


std::ostream& operator<<(std::ostream& os, const MetaInit& mi)
{
     os << "Init objectID:" << mi.objectName.ascii() << ", type:" << mi.type << ", value:" << mi.value.ascii()  ;
     return os; 
}

/**************************************************
 * MetaWire class
 *************************************************/
MetaWire::MetaWire(QString objA, QString sigName, QString objB, QString sltName)
{
     objectSender = objA;
     sender = MetaObject::getObject(objectSender);
     objectReciever = objB;
     receiver = MetaObject::getObject(objectReciever);
     signalName = sigName;
     realSignalName = SIGNAL_PREFIX + sigName;
     slotName = sltName;
     realSlotName = SLOT_PREFIX + slotName ; 
}

bool MetaWire::operator<(const MetaWire& mw)
{
	return ((objectSender + signalName + objectReciever + slotName)
		< (mw.objectSender + mw.signalName + mw.objectReciever 
			+ mw.slotName));


}

std::ostream& operator<<(std::ostream& os, const MetaWire& mw)
{
     os << "Wire sender: " << mw.objectSender.ascii() << ", signal: " << mw.signalName.ascii()
	<< ", reciever: " << mw.objectReciever.ascii() << ", slot: " << mw.slotName.ascii();
     return os; 
}


/***************************************************
 * MetaSheet class
 **************************************************/
MetaSheetList MetaSheet::msl;


MetaSheet::MetaSheet(QString id)
{
     sheetID = id;
     blockLink = false;
     //wires = NULL;
     registerSheet();
}

void MetaSheet::registerSheet()
{
     msl.insert(sheetID, this);
}

MetaSheet::~MetaSheet()
{
     msl.remove(sheetID);
}


void MetaSheet::clear()
{
     QDictIterator<MetaSheet> it=getIterator();
     delete it.current();
     for ( ; it.current() ; ++it)
	  delete it.current();

}


std::ostream& operator<<(std::ostream& os, const MetaSheet& ms)
{
     os << "Sheet name: " << ms.sheetID.ascii();
     return os; 
}


/*********************************************************
 * MetaBlock class
 ********************************************************/

// Note pour plus tard, laisser absolument ces deux choses en statique
QDict<MetaIO> MetaBlock::siglist;
QDict<MetaIO> MetaBlock::sltlist;



MetaBlock::MetaBlock(QString id, bool p) : MetaObject("MetaBlock", id, p) 
{
     instanciated = false;
     fileRepresentation = false;
     sheet = NULL;
     block = NULL;
}


void MetaBlock::addSignal(MetaIO* sig) 
{ 
     siglist.insert( sig->getPublicName() , sig ); 
}


void MetaBlock::addSlot(MetaIO* slt) 
{ 
     sltlist.insert( slt->getPublicName() , slt ); 
}


QStrList MetaBlock::getObjectSlots(QString s)
{
	QStrList res;
	QDictIterator<MetaIO> lst = getSlotList();
       	for( ; lst.current() ; ++lst)
	{
		if (lst.currentKey().startsWith(s))
			res.append(MetaObject::stripNamespace(lst.currentKey()));

	}	
	return res;

}

QStrList MetaBlock::getObjectSignals(QString s)
{
	QStrList res;
	QDictIterator<MetaIO> lst = getSignalList();
	for (; lst.current() ; ++lst)
	{
		if (lst.currentKey().startsWith(s))
			res.append(MetaObject::stripNamespace(lst.currentKey()));
	}
	return res;
}


