/*
  name: include/arcs/arcsabstractfamily.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSABSTRACTFAMILY_H__
#define __ARCSABSTRACTFAMILY_H__


#include <arcs/arcsdll.h>

#include <QStringList>

class ARCSAbstractComponent;

/*! \brief Abstract class representing a family of components i.e. a factory of factories of components.
 *
 * This class must be used when introducing a new family of components inside ARCS.
 * To implement a new family following this guideline, you should have a header like this :
 * \code
 * #ifndef __MYFAMILY_H__
 * #define __MYFAMILY_H__
 * #include <arcs/arcsabstractfamily.h>
 *
 *  class MyFamily : public ARCSAbstractFamily
 *  {
 *   public:
 *    MyFamily();
 *    virtual ~MyFamily();
 *
 *    virtual QStringList factoryList();
 *    virtual bool addFactory(QString type, ARCSAbstractComponent* cmp);
 *    virtual bool removeFactory(QString type);
 *    virtual ARCSAbstractComponent* instanciante(QString type);
 *    virtual void destroy(ARCSAbstractComponent* component);
 *    virtual QString name() const;
 *  };
 * #endif //__MYFAMILY__
 * \endcode
 *
 * \author Jean-Yves Didier
 * \date November, 2008
 * \ingroup lib
 */
class DLL_POINT ARCSAbstractFamily
{
public:
     virtual ~ARCSAbstractFamily() {}

     /*! \brief Returns a list of available factories.
      */
     virtual QStringList factoryList() = 0 ;
     /*! \brief Adds a component factory to the family.
      */
     virtual bool addFactory(QString type, ARCSAbstractComponent* cmp) = 0;
     /*! \brief Removes a component factory to the family.
      */
     virtual void removeFactory(QString type) = 0;

     /*! \brief Returns a component.
      */
     virtual ARCSAbstractComponent* instanciate(QString type) = 0;

     /*! \brief Destroys a component.
      */
     virtual void destroy(ARCSAbstractComponent* component) = 0;

     /*! \brief name of the family
      */
     virtual QString name() const=0;

     //! Describes wether the family is internal to ARCS or not.
     virtual bool isInternal() const;
};


#endif //__ARCSABSTRACTFAMILY_H__
