/*
  name: include/arcs/arcsconnection.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSCONNECTION_H__
#define __ARCSCONNECTION_H__

#include <arcs/arcsabstractcomponent.h>
#include <iostream>
#include <QPointF>
#include <QList>
#include <arcs/arcsdll.h>

//! Describes a connection
/*! This class represents a connection between two components.
 * It is also in charge of handling connections/desconnections.
 * 
 * \author Jean-Yves Didier
 * \date December, the 5th, 2008
 * \ingroup lib
 */
class DLL_POINT ARCSConnection
{
public:
     /*! Constructor by default. */
     ARCSConnection();
     /*! Copy constructor */
     ARCSConnection(const ARCSConnection & a);
     //! Constructor creating a non empty connection
     /*! \param s source component
       * \param sgn signal name
       * \param d destination component
       * \param slt slot name
       * \param q tells wether the connection is queued or not.
       */
     ARCSConnection(ARCSAbstractComponent* s, QString sgn, ARCSAbstractComponent * d, QString slt, bool q = false);

     /*! Sets the source component of the connection */
     void setSource(ARCSAbstractComponent* a) { source = a ; }
     /*! Sets the destination component of the connection */
     void setDestination(ARCSAbstractComponent* a) { destination = a; }
     /*! Sets the slot name of the connection */
     void setSlotName(QString s) { slotName = s; }
     /*! Sets the signal name of the connection */
     void setSignalName(QString s) { signalName = s; }
     //! Tells wether the connection will be queued or not.
     /*! When a connection is queued, parameters are given as events.
      * It is sometimes useful when components are triggering each other in a loop
      * since such behaviour might explode the function call stack.
      * \param b Connection is queued when set to <tt>true</tt>
      */
      void setQueued(bool b) { queued = b; }

      QList<QPointF>& getCoordinates() { return coordinates; }
      void setCoordinates(QList<QPointF> coords) {  coordinates.clear(); coordinates += coords; }

     /*! Returns the source of the connection */
     ARCSAbstractComponent* getSource() { return source; }
     /*! Returns the destination of the connection */
     ARCSAbstractComponent* getDestination() { return destination; }

     QString getSourceName() { return source->getProperty("id").toString(); }
     QString getDestinationName() { return destination->getProperty("id").toString(); }

     /*! Returns the slot name of the connection */
     QString getSlotName() { return slotName; }
     /*! Returns the signal name of the connection */
     QString getSignalName() { return signalName; }
     /*! Tells wether the connection is queued or not */
     bool isQueued() { return queued; }
     
     //! Make the connection happen.
     /*! \return <tt>true</tt> if connection succeeded, <tt>false</tt> otherwise.
       */
     bool connect() ;
     //! Disconnects the connection.
     /*! \return <tt>true</tt> if disconnection succeeded, <tt>false</tt> otherwise.
       */
     bool disconnect(); 

     /*! comparison operator */
     bool operator==(ARCSConnection c)
     {
	  return ((source == c.source) && (destination == c.destination) && 
		  (slotName == c.slotName) && (signalName == c.signalName) );
     }

     bool isComplete() { return source && destination && !slotName.isEmpty() && !signalName.isEmpty(); }

     static ARCSConnection  null;

private:
     ARCSAbstractComponent* source;
     ARCSAbstractComponent* destination;
     QString slotName;
     QString signalName;
     QList<QPointF> coordinates;
     bool queued;

};

#endif //__ARCSCONNECTION_H__
