/*
  name: include/arcs/arcssheet.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSSHEET_H__
#define __ARCSSHEET_H__

#include <arcs/arcscontext.h>
#include <arcs/arcsconnection.h>
#include <arcs/arcsinit.h>
#include <arcs/arcsdll.h>

#include <QMutex>
#include <QHash>
#include <QSet>
#include <iostream>

//! Maintains connections between objects.
/*! 
 * \author Jean-Yves Didier
 * \ingroup lib
 * \date December, the 5th, 2008.
 */
class DLL_POINT ARCSSheet
{
public:

     //! Constructor
     ARCSSheet(ARCSContext* ctx=0, QString n=QString::null);
     //! Copy constructor
     /*! <b>Warning:</b> When a sheet is copied, the context it depends on is still the same !
      */
     ARCSSheet(const ARCSSheet & sheet);
     
     //! Adds a connection to the sheet structure
     /*! \param src source component's Id
      * \param sgn source component's signal
      * \param dst destination component's Id
      * \param slt destination component's slot
      * \param q tells wether the connection is queued or not (false by default)
      * \param head tells wether the connection should be inserted on top of list or not (false by default)
      */
     ARCSConnection& addConnection(QString src, QString sgn, QString dst, QString slt, bool q = false, bool head=false);


     //! Adds a pre-connection initialisation to the sheet structure
     /*! \param dst destination component's id 
      * \param slt destination component's slot
      * \param t initialisation type
      * \param val initialisation value
      */
     ARCSInit& addPreconnect(QString dst, QString slt, QString t, QString val)
     {
        return createInit(preconnects, dst, slt, t, val);
     }

     //! Adds a post-connection initialisation to the sheet structure
     /*! \param dst destination component id
      * \param slt destination component slot
      * \param t initialisation type
      * \param val initialisation value
      */
     ARCSInit& addPostconnect(QString dst, QString slt, QString t, QString val)
     {
	  return createInit(postconnects, dst, slt, t, val);
     }

     //! Adds a cleanup invocation to the sheet structure
     /*! \param dst destination component id
      * \param slt destination component slot
      * \param t initialisation type
      * \param val initialisation value
      */
     ARCSInit& addCleanup(QString dst, QString slt, QString t, QString val)
     {
	  return createInit(cleanups, dst, slt, t, val);
     }


     //! Allows to retrieve connections
     /*! \param sources list of source component ids to fill
       * \param signls list of signals to fill
       * \param destinations list of destination component ids to fill
       * \param slts list of slots to fill
       */
     void getConnections(QStringList & sources, QStringList & signls, QStringList & destinations, QStringList & slts);

     //! returns a reference to a connection in the sheet structure
     /*! \param src source component's Id
      * \param sgn source component's signal
      * \param dst destination component's Id
      * \param slt destination component's slot
      */
     ARCSConnection & getConnection(QString source, QString sgn, QString dst, QString slt);


     int getConnectionIndex(ARCSConnection& connection) { return connections.indexOf(connection);}
     int getConnectionIndex(QString source, QString sgn, QString dst, QString slt);

     int getPreconnectIndex(ARCSInit& init) { return preconnects.indexOf(init);}
     int getPostconnectIndex(ARCSInit& init) { return postconnects.indexOf(init); }
     int getCleanupIndex(ARCSInit& init) { return cleanups.indexOf(init); }


     ARCSInit& getPreConnect(QString dst,QString slt, QString t, QString val)
     {
         return getInit(preconnects,dst,slt,t,val);
     }

     ARCSInit& getPostConnect(QString dst,QString slt, QString t, QString val)
     {
         return getInit(postconnects,dst,slt,t,val);
     }

     ARCSInit& getCleanup(QString dst,QString slt, QString t, QString val)
     {
         return getInit(cleanups,dst,slt,t,val);
     }


     void swapConnections(int x,int y);


     //! Allows to retrieve preconnection invocations
     /*! \param destinations list of destination component ids to fill
       * \param slts list of slots to fill
       * \param types list of invocation type
       * \param values list of values
       */
     void getPreconnects(QStringList & destinations, QStringList & slts, QStringList & types, QStringList & values)
     {
	  getInits(preconnects, destinations, slts, types, values);
     }

     //! Allows to retrieve postconnection invocations
     /*! \param destinations list of destination component ids to fill
       * \param slts list of slots to fill
       * \param types list of invocation type
       * \param values list of values
       */
     void getPostconnects(QStringList & destinations, QStringList & slts, QStringList & types, QStringList & values)
     {
	  getInits(postconnects, destinations, slts, types, values);
     }

     //! Allows to retrieve cleanup invocations
     /*! \param destinations list of destination component ids to fill
       * \param slts list of slots to fill
       * \param types list of invocation type
       * \param values list of values
       */
     void getCleanups(QStringList & destinations, QStringList & slts, QStringList & types, QStringList & values)
     {
	  getInits(cleanups, destinations, slts, types, values);
     }

     /*! Activates a sheet i.e. enables preconnections, connections and postconnections */
     void activate() { mutex->lock(); preconnect(); connect(); activated = true; mutex->unlock(); postconnect(); }
     /*! Deactivates a sheet i.e. disconnect all and perform cleanups */
     void deactivate() { mutex->lock(); disconnect(); cleanup(); activated = false; mutex->unlock();}

     void removeConnection(ARCSConnection c) { connections.removeAll(c); }
     void removePreconnect(ARCSInit i) { preconnects.removeAll(i); }
     void removePostconnect(ARCSInit i) { postconnects.removeAll(i); }
     void removeCleanup(ARCSInit i) { cleanups.removeAll(i); }
     ARCSContext* getContext() { return context; }
     void setContext(ARCSContext* ctx) { context = ctx; }


     /*! Performs preconnections */
     void preconnect() { batchInit(preconnects); }
     /*! Connects all links */
     void connect(); 
     /*! Performs postconnections */
     void postconnect()  { batchInit(postconnects); }
     /*! Disconnect all */
     void disconnect();
     /*! Performs some cleanup */
     void cleanup() { batchInit(cleanups); }

     void reset();
     void resetConnections() { connections.clear(); }
     void resetPreconnects() { preconnects.clear(); }
     void resetPostconnects() { postconnects.clear(); }
     void resetCleanups() { cleanups.clear(); }
     void resetProperties() { properties.clear(); }

     // Implementing properties
     /*! Set a property */
     void setProperty(QString name,QString value) { properties[name] = value ; }
     /*! Get a property */
     QString getProperty(QString name) {
           if (properties.contains(name))
               return properties[name];
           else
               return QString();
     }

     void removeProperty(QString name) {
         properties.remove(name);
     }


     //! Retrieves the list of components that are used in this sheet.
     /*! The list of components is given by inspecting connections and invocations
      */
     QStringList getComponentList();

     void removeComponent(QString name);


     /*! Return the list of properties stored in the sheet */
     QStringList getPropertyList() { return properties.keys(); }


     /*! Checks wether the sheet is activated or not */
     bool isActivated() { bool r; mutex->lock(); r = activated; mutex->unlock(); return r; }

     /*! Sets the name of the sheet */
     void setName(QString s) { name = s; }
     /*! returns the name of the sheet */
     QString getName() { return name;}

     typedef ARCSInit& (ARCSSheet::*AddInvocationMethod) (QString, QString, QString, QString);
     typedef void (ARCSSheet::*GetInvocationsMethod) (QStringList& , QStringList & , QStringList &, QStringList&);

     bool isEmpty() { return connections.isEmpty() || preconnects.isEmpty() || postconnects.isEmpty() || cleanups.isEmpty(); }
     
     static ARCSSheet null;


     void swapPreconnects(int x, int y) { swapInits(preconnects,x,y); }
     void swapPostconnects(int x,int y) { swapInits(postconnects,x,y); }
     void swapCleanups(int x,int y) { swapInits(cleanups,x,y); }

     void swapInits(QList<ARCSInit> &lst, int x, int y);



private:
     void removeInitWithComponent(QList<ARCSInit> &lst, QString name);
     void listComponentInit(QList<ARCSInit>& lst, QSet<QString>& set);
     void batchInit(QList<ARCSInit> & lst);
     ARCSInit& createInit(QList<ARCSInit> & lst, QString dst, QString slt, QString t, QString val);     
     ARCSInit& getInit(QList<ARCSInit> &lst, QString dst, QString slt, QString t, QString val);
     void getInits(QList<ARCSInit> &lst, QStringList & destinations, 
		   QStringList & slots, QStringList & types, QStringList& values);

     ARCSContext* context;
     QList<ARCSConnection> connections;
     QList<ARCSInit> preconnects;
     QList<ARCSInit> postconnects;
     QList<ARCSInit> cleanups;
     QString name;
     QMutex* mutex;

     QHash<QString,QString> properties;

     bool activated;
};


#endif //__ARCSSHEET_H__
