/*
  name: lib/arcsabstractcomponent.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcsabstractcomponent.h>
#include <arcs/arcsqdynamicobject.h>
#include <arcs/arcsfactory.h>
#include <QtDebug>
#include <iostream>
#include <QFile>
#include <QTextStream>
#include <QMetaObject>


const QString ARCSAbstractComponent::slotPrefix = "1" ;
const QString ARCSAbstractComponent::signalPrefix = "2" ;


ARCSAbstractComponent::ARCSAbstractComponent()
{
  instanciated =false;
}

ARCSAbstractComponent::~ARCSAbstractComponent()
{
}


ARCSAbstractComponent* ARCSAbstractComponent::clone()
{
     ARCSAbstractComponent* res = ARCSFactory::getInstance()->createComponent(this->getType());
     res->parseString(this->toString());

     if (res == 0)
         return res;
     
     res->setType(this->getType());
     res->setFamily(this->getFamily());
     
     QStringList lst = getProperties();

     for (int i = 0; i < lst.count(); i++)
     {
	  res->setProperty(lst[i], this->getProperty(lst[i]));
     }
     return res;
}


bool ARCSAbstractComponent::loadFile(QString fn)
{
     QFile file(fn);

     if (! file.open(QIODevice::ReadOnly))
          return false;

     QTextStream ts(&file);
     QString contents = ts.readAll();
     if (!parseString(contents))
          return false;

     file.close();
     return true;
}

bool ARCSAbstractComponent::saveFile(QString fn)
{
     QFile file(fn);
     if (!file.open(QIODevice::WriteOnly))
          return false;

     QTextStream ts(&file);
     QString contents = toString();
     ts << contents ;
     ts.flush();

     file.close();
     return true;
}


void ARCSAbstractComponent::setProperty(QString name, QVariant value)
{
     properties.insert(name, value);
}


QVariant ARCSAbstractComponent::getProperty(QString name)
{
     QVariant prop;

     if (properties.contains(name))
	  prop = properties[name];

     return prop;
}


void ARCSAbstractComponent::removeProperty(QString name)
{
     properties.remove(name);
}


void ARCSAbstractComponent::resetProperties()
{
     properties.clear();
}


QStringList ARCSAbstractComponent::getProperties()
{
     return properties.keys();
}


bool ARCSAbstractComponent::init(QString slt, QVariant* var)
{
     if (!isInstanciated())
          return false;

     QObjectList sources;
     QStringList slotNames;
     QString realName;
     this->getProxySlot(slt, sources, slotNames);

     if (sources.isEmpty())
     {
          return false;
     }
     
     bool res = true;

     for (int i=0; i < sources.size(); i++)
     {
          realName = slotNames[i].left(slotNames[i].indexOf('('));
          // insrer des choses ici
          ARCSQDynamicObject* ado = dynamic_cast<ARCSQDynamicObject*>(sources[i]);
          if (ado)
          {
               int id = ado->prepareSlotConnect(slotNames[i], slotNames[i],QString::null, slotNames[i], false);
               void * args[2];
               args[0] = 0;
               args[1] = var->data();

               ado->qt_metacall(QMetaObject::InvokeMetaMethod, id, args);
               ado->cleanSlotConnect(slotNames[i], slotNames[i], QString::null, slotNames[i]);
               res = res && true;

          }
          else
          {
               res = res && QMetaObject::invokeMethod(sources[i], qPrintable(realName),
                                                      Qt::DirectConnection,
                                                      QGenericArgument((var->typeName()),(var->data())) );
          }
     }
     return res;

}

//! \todo bug probable sur slotNames qui devrait tre indic sur j !!!!
bool ARCSAbstractComponent::connect(QString sig, ARCSAbstractComponent* dst, QString slt, bool queued)
{
     /* The lines below are a twisted hack around scripts */
     /*if (this->getType() == "Script")
          return genuineConnect(sig,dst,slt, false);
     if (dst->getType() == "Script" )
          return dst->genuineConnect(sig, this ,slt, true);*/
     /* end of twisted hack */

     if ( this->getFamily() == dst->getFamily() )
          if (genuineConnect(sig, dst, slt, queued))
               return true;

     ObjectList sources;
     ObjectList destinations;
     QStringList signalNames;
     QStringList slotNames;

     dst->getProxySlot(slt,destinations, slotNames);
     this->getProxySignal(sig, sources, signalNames);
 
     if (sources.isEmpty() || destinations.isEmpty() )
	  return false;

     bool res = true;
     int i,j;
     Qt::ConnectionType contype = queued? Qt::QueuedConnection: Qt::DirectConnection ;
     
     for (i = 0; i < sources.size(); i++)
     {
          for (j = 0; j < destinations.size(); j++)
          {
               ARCSQDynamicObject* dynSrc = dynamic_cast<ARCSQDynamicObject*>(sources[i]);
               ARCSQDynamicObject* dynDst = dynamic_cast<ARCSQDynamicObject*>(destinations[j]);

               if (dynSrc || dynDst)
               {
                    int dynSltId ;
                    int dynSigId ;

                    if (dynSrc)
                    {
                         dynSigId = dynSrc->prepareSignalConnect(signalNames[i], slotNames[j] , dst->getProperty("id").toString(), slt);
                    }
                    else
                    {
                         const QMetaObject* base = sources[i]->metaObject();
                         dynSigId = base->indexOfSignal(base->normalizedSignature(qPrintable(signalNames[i])));
                    }

                    if (dynDst)
                    {
                         dynSltId = dynDst->prepareSlotConnect(signalNames[i], slotNames[j], this->getProperty("id").toString(), sig );
                    }
                    else
                    {
                         const QMetaObject* base = destinations[j]->metaObject();
                         dynSltId = base->indexOfSlot(base->normalizedSignature(qPrintable(slotNames[j])));
                    }

                    if ( dynSigId == -1 || dynSltId == -1 )
                         return false;

                    res  = res && QMetaObject::connect(sources[i], dynSigId, destinations[j], dynSltId, contype );

               }
               else
               res = res && QObject::connect(sources[i], qPrintable(signalPrefix + signalNames[i]),
                                             destinations[j], qPrintable(slotPrefix + slotNames[j]),
                                             contype);
          }
     }
     return res;     
}


bool ARCSAbstractComponent::disconnect(QString sig, ARCSAbstractComponent* dst, QString slt)
{
     /* The lines below are a twisted hack around scripts */
     // this may not be needed anymore
     /*if (this->getType() == "Script")
     {
          this->setProperty("destination", false);
          return genuineDisconnect(sig,dst,slt);
     }
     if (dst->getType() == "Script" )
     {
          this->setProperty("destination", true);
          return dst->genuineDisconnect(sig, this ,slt);
     }*/
     /* end of twisted hack */

     if (genuineDisconnect(sig, dst, slt))
          return true;


     ObjectList sources;
     ObjectList destinations;
     QStringList signalNames;
     QStringList slotNames;

     dst->getProxySlot(slt,destinations, slotNames);
     this->getProxySignal(sig, sources, signalNames);
 
     if (sources.isEmpty() || destinations.isEmpty() )
	  return false;

     bool res = true;
     int i,j;
     
     for (i = 0; i < sources.size(); i++)
     {
          for (j = 0; j < destinations.size(); j++)
          {
               ARCSQDynamicObject* dynSrc = dynamic_cast<ARCSQDynamicObject*>(sources[i]);
               ARCSQDynamicObject* dynDst = dynamic_cast<ARCSQDynamicObject*>(destinations[j]);

               if (dynSrc || dynDst)
               {
                    int dynSltId ;
                    int dynSigId ;

                    if (dynSrc)
                    {
                         dynSigId = dynSrc->cleanSignalConnect(signalNames[i], slotNames[j], this->getProperty("id").toString(), slt );
                    }
                    else
                    {
                         const QMetaObject* base = sources[i]->metaObject();
                         dynSigId = base->indexOfSignal(base->normalizedSignature(qPrintable(signalNames[i])));
                    }

                    if (dynDst)
                    {
                         dynSltId = dynDst->cleanSlotConnect(signalNames[i], slotNames[j], dst->getProperty("id").toString(), sig );
                    }
                    else
                    {
                         const QMetaObject* base = destinations[j]->metaObject();
                         dynSltId = base->indexOfSlot(base->normalizedSignature(qPrintable(slotNames[j])));
                    }

                    if ( dynSigId == -1 || dynSltId == -1 )
                         return false;

                    res  = res && QMetaObject::disconnect(sources[i], dynSigId, destinations[j], dynSltId );

               }
               else
               {
               res = res && QObject::disconnect(sources[i], qPrintable(signalPrefix + signalNames[i]),
                                                destinations[j], qPrintable(slotPrefix + slotNames[j]));
               }
          }
     }
     return res;     
}


QStringList ARCS::getMethodList(const QMetaObject *object, QMetaMethod::MethodType type)
{
    QStringList answer;

    for (int i=0; i < object->methodCount(); i++)
    {
        QMetaMethod mmo = object->method(i);
        if (mmo.methodType() == type)
#if QT_VERSION >= 0x050000
            answer << mmo.methodSignature();
#else
            answer << mmo.signature();
#endif
    }
    return answer;
}
