/*
  name: lib/arcscontext.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcscontext.h>
//#include <arcs/arcsfactory.h>
#include <arcs/arcslog.h>
#include <QtDebug>


ARCSContext::ARCSContext()
{
     factory = ARCSFactory::getInstance();
     instanciated = false;
}


ARCSContext::ARCSContext(const ARCSContext& ctx)
{
     this->factory = ctx.factory;

     QHashIterator<QString, ARCSAbstractComponent*> i(ctx.componentPool);
     while(i.hasNext())
     {
	  i.next();
	  if (i.value())
	    componentPool[i.key()] = i.value()->clone();
     }

     constantPool = ctx.constantPool;
}



ARCSContext::~ARCSContext()
{


}

QStringList ARCSContext::computeLibraryList()
{
     int i;

     QStringList components = componentPool.keys();
     QStringList constants = constantPool.keys();
     QStringList result;
     QString lib;

     for (i=0; i < components.count(); i++)
     {
      lib = factory->getComponentLibrary(componentPool[components[i]]->getType());
      if (! lib.isEmpty())
	       if (!result.contains(lib))
		    result << lib;
     }

     for (i=0; i < constants.count(); i++)
     {
         //! \todo something not trivial is to do below in order to recover the good family.
      lib = factory->getTypeLibrary(constants[i]);
      if (! lib.isEmpty())
	       if (!result.contains(lib))
		    result << lib;
     }

     return result;
}


QStringList ARCSContext::getConstantList()
{
     return constantPool.keys();
}



QStringList ARCSContext::getComponentList()
{
     return componentPool.keys();
}


bool ARCSContext::instanciatePool()
{
     bool res = true;

     QHashIterator<QString, ARCSAbstractComponent*> i(componentPool);
     while (i.hasNext())
     {
	  i.next();
	  ARCSAbstractComponent* cmp = i.value();
	  QVariant v = cmp->getProperty("persistent");
       if (v.isValid() && i.key() != "this" )
	  {
	       if (v == QVariant(true))
		    if (!cmp->isInstanciated())
			 res = res || cmp->instanciate();
	  }
	  else
	       if (!cmp->isInstanciated())
		   res = res && cmp->instanciate();
     }
     instanciated = res;
     return res;
}



void ARCSContext::reset()
{
     constantPool.clear();

     QHashIterator<QString, ARCSAbstractComponent*> i(componentPool);

     while(i.hasNext())
     {
	  i.next();
	  factory->destroyComponent(i.value());
     }
     componentPool.clear();
}


ARCSAbstractComponent* ARCSContext::createComponent(QString name, QString className)
{
     ARCSAbstractComponent* candidate= factory->createComponent(className);
     if (candidate == 0)
     {
         ARCSLog::logCritical(ARCS_SOURCE,"cannot create component "+name+" of type "+className);
          return 0;
     }

     candidate->setProperty("id", name);

     if ( componentPool.contains(name))
     {
       qCritical() << "[Context] Component pool already contains a component named" << name;
       return 0;
     }

     componentPool.insert(name, candidate);
     return candidate;
}


bool ARCSContext::addComponent(QString name, QString className, QString contents)
{
     ARCSAbstractComponent* candidate = factory->createComponent(className);

     if (candidate == 0)
	  return false;
	       
     candidate->setProperty("id", name);
     if ( ! candidate->parseString(contents) )
	  qCritical() << "[Context] Unable to parse contents:" << contents << "for object" << name << "of type" << className ;

     if ( componentPool.contains(name))
     {
	  qCritical() << "[Context] Component pool already contains a component named" << name;
	  return false;
     }

     componentPool.insert(name, candidate);
     return true;
}

bool ARCSContext::renameComponent(QString oldName, QString newName)
{
    if (componentPool.contains(newName))
        return false;

    if (! componentPool.contains(oldName))
        return false;

    ARCSAbstractComponent* componentToRename = componentPool.take(oldName);
    componentToRename->setProperty("id",newName);
    componentPool.insert(newName,componentToRename);
    return true;
}


void ARCSContext::removeComponent(QString name)
{
     if (componentPool.contains(name))
     {
	  ARCSAbstractComponent* removal = componentPool.take(name);
	  factory->destroyComponent(removal);
     }
}


ARCSAbstractComponent* ARCSContext::getComponent(QString name)
{
     if (componentPool.contains(name))
	  return componentPool[name];
     return 0;
}


bool ARCSContext::addConstant(QString name, QString type, QString representation)
{
     if (constantPool.contains(name))
     {
	  qCritical() << "[Context] Constant pool already contains a constant name" << name;
	  return false;
     }

     QVariant var = factory->dataDeserialize(type, representation);
     
     if (!var.isValid())
	  return false;

     constantPool[name] = var;
     return true;
}


void ARCSContext::removeConstant(QString name)
{
     constantPool.remove(name);
}


QVariant ARCSContext::getConstant(QString name)
{
     if (constantPool.contains(name))
	  return constantPool[name];
     return QVariant();
}


QVariant ARCSContext::createValue(QString type, QString representation)
{
     return factory->dataDeserialize(type, representation);
}


bool ARCSContext::serializeConstant(QString name, QString & type, QString & representation)
{
     QVariant var = getConstant(name);
     if (! var.isValid())
	  return false;

     type = factory->getVariantType(var);
     representation = factory->dataSerialize(var);
     return true;
}


bool ARCSContext::modifyConstant(QString name, QString representation)
{
     QString type;
     QVariant var = getConstant(name);
     if (! var.isValid())
	  return false;

     type = factory->getVariantType(var);
     removeConstant(name);
     return addConstant(name, type, representation);
}
