/*
  name: lib/arcsgenerallogger.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcsgenerallogger.h>


#include <arcs/arcsfactory.h>
#include <iostream>


void ARCSGeneralLoggerBase::setFilename(QString s)
{
     if (logFile.is_open())
          logFile.close();

     logFile.open(qPrintable(s));
}


bool ARCSGeneralLoggerBase::canLog()
{
     if (!logFile.is_open())
          logFile.open("arcs_log.log");

     return logFile.is_open();
}


/************************************************************
  *
  ***********************************************************/

ARCSGeneralLogger::ARCSGeneralLogger(QObject* parent) : ARCSGeneralLoggerBase(parent)
{
     idx = 0;
}


QStringList ARCSGeneralLogger::getSlotList()
{
     QStringList sl ;
     sl << "log()" ;
     return sl;
}

QStringList ARCSGeneralLogger::getSignalList()
{
     return QStringList();
}


int ARCSGeneralLogger::cleanSlotConnect(QString sigName, QString /*sltName*/, QString objectName, QString actualSignal)
{
     QString signature = objectName + "." + actualSignal + "." + sigName ;

     int id = -1;
     if (map.contains(signature))
          id = map[signature];
     else
          return -1;

     map.take(signature);
     invertMap.take(id);

     return id;
}


int ARCSGeneralLogger::cleanSignalConnect(QString /*sigName*/, QString /*sltName*/, QString /*objectName*/, QString /*actualSlot*/)
{ return -1; }

int ARCSGeneralLogger::prepareSlotConnect(QString sigName, QString sltName, QString objectName, QString actualSignal, bool simulate)
{
     if (simulate)
          return 0;

     if (metaObject()->indexOfSlot(qPrintable(sltName)) != -1)
     {
          return metaObject()->indexOfSlot(qPrintable(sltName));
     }

     // first parse sigName
     QString signature = objectName + "." + actualSignal + "." + sigName ;
     LogDetails ld(signature);

     QStringList paramTypes = sigName.section("(",1,1).section(")", 0,0).split(",");

     for (int i=0; i < paramTypes.count(); i++)
          ld.addType(QMetaType::type(qPrintable(paramTypes.at(i))));

     map.insert(signature,idx);
     invertMap.insert(idx,ld);
     idx++;

     return idx-1 + metaObject()->methodCount();
}

int ARCSGeneralLogger::prepareSignalConnect(QString /*sigName*/, QString /*sltName*/, QString /*objectName*/, QString /*actualSlot*/, bool /*simulate*/)
{
    // nothing to implement here.
     return -1 ;
}

int ARCSGeneralLogger::qt_metacall(QMetaObject::Call call, int id, void ** arguments)
{
     QMutexLocker lock(&mutex);
     id = ARCSGeneralLoggerBase::qt_metacall(call,id, arguments);
     if (id < 0 || call != QMetaObject::InvokeMetaMethod)
          return id;


     if (!canLog())
     {
          std::cerr << "Log file not opened" << std::endl;
          return  -1;
     }

     // the piece of code below should be used in something else if required.

     if (!invertMap.contains(id))
     {
          std::cerr << "Map does not contain id " << id << std::endl;
          return -1;
     }

      LogDetails ld = invertMap[id];


     for (int i =0; i < ld.count(); i++)
     {
          int it = ld.getArgument(i);

          if (it != QMetaType::Void)
          {
               QVariant v(it, arguments[i+1]);
               QString s = ARCSFactory::getInstance()->dataSerialize(v);
               logFile << qPrintable(s) << qPrintable(separator) ;
          }
     }

     return -1;
}



/**********************************************************************************
 *  inner class LogDetails of ARCSSensorLogger
 **********************************************************************************/


ARCSGeneralLogger::LogDetails::LogDetails(const LogDetails& ld)
{
     signature = ld.signature;
     typeIds = ld.typeIds;
}

ARCSGeneralLogger::LogDetails::LogDetails(QString sn)
{
     signature = sn ;
}


