/*
  name: tools/editor/graphicslinkitem.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "graphicscomponentitem.h"
#include "graphicslinkitem.h"
#include "graphicsportitem.h"
#include "sheetview.h"


#include <iostream>
#include <arcs/arcsabstractcomponent.h>
#include <QBrush>
#include <QKeyEvent>
#include <QGraphicsScene>
#include <QGraphicsSceneMoveEvent>
#include <QStyle>
#include <QStyleOptionGraphicsItem>

#ifndef MAX
#define MAX(A,B) (((A)>(B))?(A):(B))
#endif

#ifndef MIN
#define MIN(A,B) (((A)<(B))?(A):(B))
#endif


GraphicsLinkItem::GraphicsLinkItem(QGraphicsItem *parent, ARCSConnection* c) : QGraphicsPathItem(parent),connection(c)
{
    int i;
    setAcceptHoverEvents(true);
    setFlag(ItemIsSelectable,true);
    markedForDeletion = false;

    source = 0;
    destination = 0;
    finalized = connection;
    defaultPen = pen();

    for(i=0; i < 4; i++)
    {
        grips[i] = new GraphicsGripLinkItem(this);
        grips[i]->hide();
        if (i>0)
        {
            grips[i]->setPrevious(grips[i-1]);
            grips[i-1]->setNext(grips[i]);
        }
    }


    if (finalized)
    {
        QList<QPointF>& coords = connection->getCoordinates();
        if (coords.count() >= 4)
        {
            for(i=0;i<4; i++)
                grips[i]->setPos(coords[i]);
        }
    }

}

GraphicsLinkItem::~GraphicsLinkItem()
{
    setSelected(false);

     // here we should access to each portItem to remove myself from their list
    // nothing to do: just remove so that the parentItem does not access to this ?
    if (destination)
        destination->removeLinkItem(this);
    if (source)
        source->removeLinkItem(this);


    // then we should remove my connection counterpart
    //! \todo we have a possible problem: should a link always delete its counterpart ? Yes if it is the user intent, no if it was a cleanup operation.

    if ( markedForDeletion )
    {
        SheetView* view = dynamic_cast<SheetView*>(scene()->views()[0]);
        if (!view)
            return ;

        if (! connection )
            return ;

        view->getSheet().removeConnection(*connection);
    }
}


void GraphicsLinkItem::redraw(bool isSource)
{
    // this is called when source or destination have moved.
    // maybe we should differenciate according to which target has moved.

    if ( ! (source && destination) )
        return ;

    /*QPointF orig = mapFromScene(source->scenePos());
    QPointF dest = mapFromScene(destination->scenePos());
    Configuration config = computeConfiguration(orig,dest);

    if (config != configuration)
    {
        computePath(orig,dest);
        return ;
    }

    if (isSource || destination == source)
    {
        grips[0]->setPos(grips[0]->x(), orig.y()) ;//orig + QPoint(20,0));

        // then grip 0 and grip 1 should move
        // move on y direction anyway for grip1

        grips[1]->setPos(grips[1]->x(),grips[0]->y());

        if (configuration != Step)
            grips[1]->setPos(grips[0]->x(),grips[1]->y());
    }
    if (!isSource || destination == source)
    {
        grips[3]->setPos(grips[3]->x(), dest.y()) ; //dest + QPoint(-20,0));
        // grip 2 and grip 3 should move
        // same reasoning as the previous one.

        grips[2]->setPos(grips[2]->x(), grips[2]->y());

        if (configuration != Step)
            grips[2]->setPos(grips[3]->x(), grips[2]->y());
        // idem
    }

    connectGrips();*/
    // this is a complete recomputation
    if (isSource || connection->getCoordinates().count() < 4)
        computePath(mapFromScene(source->scenePos()), mapFromScene(destination->scenePos()));
    else
        connectGrips();
}


void GraphicsLinkItem::setCurrentPoint(QPointF p)
{
    if (  ! (source || destination) ) return ;


    if (source)
        computePath(QPointF(0,0),p);
    else
        computePath(p,QPoint(0,0));
}


GraphicsLinkItem::Configuration GraphicsLinkItem::computeConfiguration(QPointF orig, QPointF dest)
{
    if (dest.x() - orig.x() > 40)
        return  Step;

    if (destination && source)
    {
        QRectF dstRect = mapFromScene(destination->parentItem()->sceneBoundingRect()).boundingRect();
        QRectF srcRect  = mapFromScene(source->parentItem()->sceneBoundingRect()).boundingRect();

        if (  (srcRect.y() < dstRect.y()) && (dstRect.y() < srcRect.y() + srcRect.height()) )
            return Loop;
        if (  (dstRect.y() < srcRect.y()) && (srcRect.y() < dstRect.y() + dstRect.height()) )
           return Loop;
    }
    else
    {
        QRectF parentRect = mapFromScene(parentItem()->parentItem()->sceneBoundingRect()).boundingRect();

        if ( (parentRect.y() < orig.y()) && (orig.y() < parentRect.y() + parentRect.height()) )
            return Loop;
    }
    return  S_Shape;
}


void GraphicsLinkItem::computePath(QPointF orig, QPointF dest)
{
    Configuration config = computeConfiguration(orig,dest);

    grips[0]->setPos(orig + QPoint(20,0));
    grips[3]->setPos(dest + QPointF(-20,0));

    switch(config)
    {
    case Step:
        grips[1]->setPos((grips[0]->x()+grips[3]->x())/2, grips[0]->y());
        grips[2]->setPos(grips[1]->x(), grips[3]->y());
        break;
    case S_Shape:
        grips[1]->setPos(grips[0]->x(),(grips[0]->y()+grips[3]->y())/2);
        grips[2]->setPos(grips[3]->x(),grips[1]->y());
        break;
    case Loop:
    {
        int loopHeight = 0;
        if (destination && source)
        {
            QRectF dstRect = mapFromScene(destination->parentItem()->sceneBoundingRect()).boundingRect();
            QRectF srcRect  = mapFromScene(source->parentItem()->sceneBoundingRect()).boundingRect();
            loopHeight = MAX(srcRect.y(),dstRect.y()) + MAX(srcRect.height(),dstRect.height()) - orig.y() + 20 ;
          }
        else
        {
            QRectF parentRect = mapFromScene(parentItem()->parentItem()->sceneBoundingRect()).boundingRect();
            loopHeight = parentRect.y() + parentRect.height() - orig.y() + 20 ;
        }
        grips[1]->setPos(grips[0]->x(),grips[0]->y()+loopHeight);
        grips[2]->setPos(grips[3]->x(),grips[1]->y());
    }
    }
    configuration = config;
    connectGrips(orig,dest);
}


void GraphicsLinkItem::hoverEnterEvent(QGraphicsSceneHoverEvent */*event*/)
{
    source->showName();
    destination->showName();

    for (int i=0; i < 4; i++)
        grips[i]->show();
}

void GraphicsLinkItem::hoverLeaveEvent(QGraphicsSceneHoverEvent */*event*/)
{
    source->hideName();
    destination->hideName();
    for (int i=0; i < 4; i++)
        grips[i]->hide();
}


// this is defined in order to override Qt's default behaviour for selected items.
void GraphicsLinkItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    QStyleOptionGraphicsItem myOptions(*option);
    if (myOptions.state & (QStyle::State_Selected))
    {
        QPen myPen(Qt::darkRed);
        myPen.setWidth(2);
        setPen(myPen);
        myOptions.state = myOptions.state ^ QStyle::State_Selected  ;
    }
    else
    {
        setPen(defaultPen);
    }
    QGraphicsPathItem::paint(painter,& myOptions,  widget );

}


QPainterPath GraphicsLinkItem::shape() const
{
    if ((! source)  || (! destination) )
        return QGraphicsPathItem::shape();

    QPointF orig = mapFromScene(source->scenePos());
    QPointF dest = mapFromScene(destination->scenePos());

    QPainterPath pp;

    QRectF rectLine(QPointF(MIN(orig.x(),grips[0]->x()),MIN(orig.y(),grips[0]->y())),
                    QPointF(MAX(orig.x(),grips[0]->x()),MAX(orig.y(),grips[0]->y())));

    pp.addRect(rectLine.adjusted(-2,-2,2,2));

    for (int i=0; i < 3;i++ )
    {
        rectLine = QRectF(QPointF(MIN(grips[i]->x(),grips[i+1]->x()), MIN(grips[i]->y(),grips[i+1]->y())),
                          QPointF(MAX(grips[i]->x(),grips[i+1]->x()), MAX(grips[i]->y(),grips[i+1]->y())));
        pp.addRect(rectLine.adjusted(-3,-3,3,3));
    }

    rectLine = QRectF(QPointF(MIN(dest.x(),grips[3]->x()),MIN(dest.y(),grips[3]->y())),
                      QPointF(MAX(dest.x(),grips[3]->x()),MAX(dest.y(),grips[3]->y())));
    pp.addRect(rectLine.adjusted(-2,-2,2,2));

    return pp;
}

void GraphicsLinkItem::connectGrips(QPointF orig, QPointF dest)
{
    QPainterPath pp ;
    pp.moveTo(orig);
    pp.lineTo(grips[0]->pos());
    pp.lineTo(grips[1]->pos());
    pp.lineTo(grips[2]->pos());
    pp.lineTo(grips[3]->pos());
    pp.lineTo(dest);
    setPath(pp);

    // here we will also update grip position
    if (connection)
    {
        QList<QPointF>& coords = connection->getCoordinates();
        coords.clear();
        for (int i=0; i< 4; i++)
            coords.append(grips[i]->pos());
        //connection->setCoordinates(coords);
    }
}


void GraphicsLinkItem::connectGrips()
{
    // retrouver orig et dest ?
    QPointF orig = mapFromScene(source->scenePos());
    QPointF dest = mapFromScene(destination->scenePos());
    connectGrips(orig,dest);
}


void GraphicsLinkItem::finalize()
{
    if (! (destination && source))
        return ;

    SheetView* view = dynamic_cast<SheetView*>(scene()->views()[0]);
    if (!view)
        return ;

    GraphicsComponentItem* sourceItem = dynamic_cast<GraphicsComponentItem*>(source->parentItem());
    GraphicsComponentItem* destinationItem = dynamic_cast<GraphicsComponentItem*>(destination->parentItem());

    if (! (sourceItem && destinationItem))
        return ;

    ARCSSheet& sheet = view->getSheet();
    connection = & (sheet.addConnection(sourceItem->getComponent()->getProperty("id").toString(),
                                     source->getName(),
                                     destinationItem->getComponent()->getProperty("id").toString(),
                                     destination->getName()));

    finalized = true;
    //setFlag(ItemClipsToShape,true);

}



/******************************************************************************
  GraphicsGripLinkItem

  ****************************************************************************/

GraphicsGripLinkItem::GraphicsGripLinkItem(QGraphicsItem *parent):
          QGraphicsRectItem(-5,-5,10,10, parent)
{

    setFlag(QGraphicsItem::ItemIsMovable,true);
    setAcceptHoverEvents(true);
    QBrush br=brush();
    br.setStyle(Qt::SolidPattern);
    br.setColor(Qt::magenta);

    setBrush(br);

    next = 0;
    previous = 0;
}


void GraphicsGripLinkItem::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    // we must look for possible movements :
    // if there is no previous or next, then moves are only possible along x
    // otherwise, moves are possible along x and y !
   /*QPointF lastPos = event->lastScenePos();
    QPointF curPos = event->scenePos();*/

    // here we compute the move and dispatch x and y moves to the right grip.
    QPointF deltaMove = parentItem()->mapFromScene(event->scenePos()) - pos();



    if (previous && next)
    {
        bool yMove =  ! ((pos().y() == next->pos().y()  && ! next->next) || (pos().y() == previous->pos().y()  && ! previous->previous));

       if (pos().x() == next->pos().x() )
            next->setPos(next->pos() + QPointF(deltaMove.x(),0) );
        else
           if (yMove)
                next->setPos(next->pos() + QPointF(0,deltaMove.y()) );


        if (pos().x() == previous->pos().x() )
            previous->setPos(previous->pos() + QPointF(deltaMove.x(),0) );
        else
            if (yMove)
                previous->setPos(previous->pos() + QPointF(0,deltaMove.y()) );

        if (yMove)
            setPos(parentItem()->mapFromScene(event->scenePos()));
        else
            setPos(pos() + QPointF(deltaMove.x(),0));
    }
    else
    {
        if (previous)
            if (previous->pos().y() != pos().y())
                previous->setPos(previous->pos() + QPointF(deltaMove.x(),0) );
        if (next)
            if (next->pos().y() != pos().y())
                next->setPos(next->pos() + QPointF(deltaMove.x(),0) );

        setPos(pos() + QPointF(deltaMove.x(),0));

    }

    dynamic_cast<GraphicsLinkItem*>(parentItem())->connectGrips();

    //setPos(parentItem()->mapFromScene(event->scenePos()));
}
