/*
  name: tools/editor/sheetview.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "sheetview.h"
#include <QDragEnterEvent>
#include <QDropEvent>
#include <QGraphicsScene>
#include <QGraphicsRectItem>
#include <QTransform>
#include <QMessageBox>

#include <QApplication>
#include <QClipboard>
#include <QMimeData>

#include <QMenu>

#include <QPen>
#include <QBrush>

#include <iostream>

#include <arcs/arcscompositecomponent.h>

#include "graphicscomponentitem.h"
#include "graphicslinkitem.h"
#include "orderconnectionsdialog.h"

/******************************************************************************
  A sheet should have something in order to store graphical details.
  The main idea :
     - list of components that are in the sheet ;
     - for each components, its coordinates and visible ports ;
     - for each connection : graphical details.


  ****************************************************************************/

#ifndef MAX
#define MAX(A,B) (((A)>(B))?(A):(B))
#endif

#ifndef MIN
#define MIN(A,B) (((A)<(B))?(A):(B))
#endif



SheetView::SheetView(ARCSSheet& s, QWidget *parent) : QGraphicsView(parent),sheet(s)
{
    setScene(new QGraphicsScene(this));
    //scene()->setSceneRect(0,0,1024,768);

    setAcceptDrops(true);
    setResizeAnchor(QGraphicsView::NoAnchor);
    setTransformationAnchor(QGraphicsView::NoAnchor);

    setSceneRect(viewport()->rect());
    setAlignment(Qt::AlignLeft|Qt::AlignTop);

    //rectSelection = new QGraphicsRectItem(0,scene());
    rectSelection = new QGraphicsRectItem();
    scene()->addItem(rectSelection);
    rectSelection->setPen(QPen(Qt::darkGreen));
    rectSelection->setBrush(QBrush(QColor(0,255,0,63)));
    rectSelection->setVisible(false);


    // this code is written in order to ease buttons and actions manipulation.
    actionSelectAll = new QAction("Select all",this);
    actionSelectAll->setShortcuts(QKeySequence::SelectAll);
    connect(actionSelectAll,SIGNAL(triggered()),this,SLOT(selectAll()));
    actionCopy = new QAction("&Copy",this);
    actionCopy->setShortcuts(QKeySequence::Copy);
    connect(actionCopy,SIGNAL(triggered()),this,SLOT(copy()));
    actionPaste = new QAction("&Paste",this);
    actionPaste->setShortcuts(QKeySequence::Paste);
    connect(actionPaste,SIGNAL(triggered()),this,SLOT(paste()));
    actionDelete = new QAction("Delete selection",this);
    actionDelete->setShortcuts(QKeySequence::Delete);
    connect(actionDelete,SIGNAL(triggered()),this,SLOT(deleteSelection()));
    QAction* sep1 = new QAction(this);
    sep1->setSeparator(true);
    QAction* sep2 = new QAction(this);
    sep2->setSeparator(true);

    actionOrderInvocations = new QAction("Order invocations...",this);
    connect(actionOrderInvocations,SIGNAL(triggered()),this,SLOT(orderInvocations()));
    actionOrderConnections = new QAction("Order connections...",this);
    connect(actionOrderConnections,SIGNAL(triggered()),this,SLOT(orderConnections()));


    addAction(actionOrderConnections);
    addAction(actionOrderInvocations);
    addAction(sep2);
    addAction(actionCopy);
    addAction(actionPaste);
    addAction(actionDelete);
    addAction(sep1);
    addAction(actionSelectAll);

    layoutSheet();

}


SheetView::~SheetView()
{
    delete scene();

}


void SheetView::layoutSheet()
{
    int i,j;

    // first we should read component-list property
    QString list = sheet.getProperty("component-list");
    QStringList actualComponents = sheet.getComponentList();


    if (list.isEmpty())
        componentList = actualComponents;
    else
        componentList = list.split(';').toSet().unite(actualComponents.toSet()).intersect(sheet.getContext()->getComponentList().toSet()).toList();


    //componentList = list.split(";");
    for (i=0; i < componentList.count(); i++)
    {
            GraphicsComponentItem* gci = new GraphicsComponentItem(sheet.getContext()->getComponent(componentList[i]));
            gci->show();
            scene()->addItem(gci);
            QString pos = sheet.getProperty(componentList[i]+"-pos");
            if (!pos.isEmpty())
                gci->setPos(pos.section(",",0,0).toFloat(),pos.section(",",1,1).toFloat());

            QString slts = sheet.getProperty(componentList[i]+"-slots");
            if (!slts.isEmpty())
            {
                QStringList slotList = slts.split(";");
                for (j=0; j < slotList.count(); j++ )
                    gci->addSlot(slotList[j],false);
            }
            QString sgnls = sheet.getProperty(componentList[i]+"-signals");
            if (!sgnls.isEmpty())
            {
                QStringList signalList = sgnls.split(";");
                for (j=0; j < signalList.count(); j++ )
                    gci->addSignal(signalList[j],false);
            }
    }
    // after all this, we should assess if all components have been rendered.
    // if not, we should compute their layout (maybe we can use dot ?)


    // below, we process connections
    QStringList sources;
    QStringList destinations;
    QStringList signalList;
    QStringList slotList;
    QStringList invocationTypes;
    QStringList invocationValues;
    sheet.getConnections(sources,signalList,destinations,slotList);
    for (i=0; i< sources.count(); i++)
    {
        ARCSConnection& c = sheet.getConnection(sources[i],signalList[i],destinations[i],slotList[i]);


        GraphicsComponentItem* srcGComponent = findComponentByName(sources[i]);
        GraphicsComponentItem* dstGComponent = findComponentByName(destinations[i]);
        if (srcGComponent && dstGComponent)
        {
            GraphicsPortItem* sgnPort = srcGComponent->getSignalPort(signalList[i]);
            if (!sgnPort)
            {
                srcGComponent->addSignal(signalList[i]);
                sgnPort = srcGComponent->getSignalPort(signalList[i]);
            }

            GraphicsPortItem* sltPort = dstGComponent->getSlotPort(slotList[i]);
            if (!sltPort)
            {
                dstGComponent->addSlot(slotList[i]);
                sltPort = dstGComponent->getSlotPort(slotList[i]);

            }

            if (sgnPort && sltPort)
            {
                GraphicsLinkItem* link = new GraphicsLinkItem(sgnPort,&c);
                link->setSource(sgnPort);
                sgnPort->addLink(link);
                link->setDestination(sltPort);
                sltPort->addLink(link);
                link->show();
                link->redraw(false);
            }
        }
    }

    //after all this, we process invocations
    sheet.getPreconnects(destinations,slotList,invocationTypes,invocationValues);
    layoutInvocations(GraphicsInvocationItem::Pre,destinations,slotList,invocationTypes,invocationValues);
    sheet.getPostconnects(destinations,slotList,invocationTypes,invocationValues);
    layoutInvocations(GraphicsInvocationItem::Post,destinations,slotList,invocationTypes,invocationValues);
    sheet.getCleanups(destinations,slotList,invocationTypes,invocationValues);
    layoutInvocations(GraphicsInvocationItem::Cleanup,destinations,slotList,invocationTypes,invocationValues);


    sheet.setProperty("component-list",componentList.join(";"));
    if (componentList.count() == 0)
        actionSelectAll->setEnabled(false);
    actionCopy->setEnabled(false);
    actionDelete->setEnabled(false);


    scene()->setSceneRect(sceneRect().united(scene()->itemsBoundingRect()));
    setSceneRect(scene()->sceneRect());
}

void SheetView::layoutInvocations(GraphicsInvocationItem::InvocationType invocationType, QStringList destinations, QStringList slotList, QStringList typeList, QStringList valueList)
{
    for (int i=0; i < destinations.count(); i++)
    {
        ARCSInit* init = 0 ; //ARCSInit::null;
        switch (invocationType)
        {
        case GraphicsInvocationItem::Pre:
            init = &sheet.getPreConnect(destinations[i],slotList[i],typeList[i],valueList[i]);
            break;
        case GraphicsInvocationItem::Post:
            init = &sheet.getPostConnect(destinations[i],slotList[i],typeList[i],valueList[i]);
            break;
        case GraphicsInvocationItem::Cleanup:
            init = &sheet.getCleanup(destinations[i],slotList[i],typeList[i],valueList[i]);
            break;
        }

        GraphicsComponentItem* dstGComponent = findComponentByName(destinations[i]);
        if (dstGComponent)
        {
            GraphicsPortItem* sltPort = dstGComponent->getSlotPort(slotList[i]);
            if (!sltPort)
            {
                dstGComponent->addSlot(slotList[i]);
                sltPort = dstGComponent->getSlotPort(slotList[i]);
            }
            if (sltPort && init )
            {
                //! \todo let's hope this line below works !
                new GraphicsInvocationItem(sltPort,invocationType,*init);
            }
        }
    }
}



void SheetView::dragEnterEvent(QDragEnterEvent *event)
{
    if (event->mimeData()->hasFormat("text/plain"))
        if (event->source()->property("appId") == this->property("appId"))
            event->acceptProposedAction();
}

void SheetView::dragMoveEvent(QDragMoveEvent *event)
{
    if (event->mimeData()->hasFormat("text/plain"))
        if (event->source()->property("appId") == this->property("appId"))
            event->acceptProposedAction();
}


void SheetView::dropEvent(QDropEvent *event)
{
    if (event->source()->property("appId") != this->property("appId"))
        return;

    addComponent( event->pos(),event->mimeData()->text());
    event->acceptProposedAction();
}



void SheetView::updateSheet()
{
    clearScene();
    layoutSheet();
}


void SheetView::clearScene()
{
    // clearing the scene is somewhat complicated.
    // it is better to do it in the right order
    int i;
    QList<QGraphicsItem*> itemList = scene()->items();
    for (i=0; i < itemList.count();i++)
    {
        if (itemList[i]->type() == GraphicsComponentItem::Type)
        {
            GraphicsComponentItem* item = dynamic_cast<GraphicsComponentItem*>(itemList[i]);
            delete item;
            // each time an item is removed from list, we must regenerate list of selected items.
            itemList = scene()->items();
            i=-1;
        }
    }


    itemList = scene()->items();
    for (i=0; i < itemList.count();i++)
    {
        if (itemList[i]->type() == GraphicsPortItem::Type)
        {
            GraphicsPortItem* item = dynamic_cast<GraphicsPortItem*>(itemList[i]);
            delete item;
            // each time an item is removed from list, we must regenerate list of selected items.
            itemList = scene()->items();
            i=-1;
        }
    }

    itemList = scene()->items();
    for (i=0; i < itemList.count(); i++)
    {
        if (itemList[i]->type() == GraphicsInvocationItem::Type)
        {
            GraphicsInvocationItem* item = dynamic_cast<GraphicsInvocationItem*>(itemList[i]);
            delete item;
            // each time an item is removed from list, we must regenerate list of selected items.
            itemList = scene()->items();
            i=-1;

        }
        else
        {
            if (itemList[i]->type() == GraphicsLinkItem::Type )
            {
                GraphicsLinkItem* item = dynamic_cast<GraphicsLinkItem*>(itemList[i]);
                delete item;
                // each time an item is removed from list, we must regenerate list of selected items.
                itemList = scene()->items();
                i=-1;
            }
        }

    }
    //scene()->clear();

    actionSelectAll->setEnabled(false);
    actionCopy->setEnabled(false);
}



void SheetView::addComponent(QPoint p, QString name)
{
    ARCSContext* ctx=sheet.getContext();
    if (!ctx )
        return ;

    if (!scene())
        return ;

    QStringList sl = ctx->getComponentList();

    if (!ctx->getComponent(name))
    {
        std::cerr << "component not found" << std::endl;
        return ;
    }

    // here we should also check if the component is already in the sheet or not.
    // therefore sheets should have data on components that are inside them.

    if (componentList.contains(name))
        return ;

    componentList.append(name);
    sheet.setProperty("component-list",componentList.join(";"));

    // now let's draw our component.
    GraphicsComponentItem* gci = new GraphicsComponentItem(ctx->getComponent(name));
    QTransform tr = transform();

    gci->setPos(mapToScene(tr.map(p)));
    gci->show();
    scene()->addItem(gci);
    //! \todo maybe the instruction below should be handled by GraphicsComponentItem
    sheet.setProperty(name+"-pos",QString::number(gci->pos().x())+","+QString::number(gci->pos().y()));

    scene()->setSceneRect(sceneRect().united(scene()->itemsBoundingRect()));
    setSceneRect(scene()->sceneRect());
    actionSelectAll->setEnabled(true);
}



GraphicsComponentItem* SheetView::findComponentByName(QString name)
{
    //! \todo this may not be optimal.
    QList<QGraphicsItem*> items = scene()->items();
    for (int i=0; i<items.count(); i++)
    {
        if (items[i]->type() == GraphicsComponentItem::Type)
        {
            GraphicsComponentItem* gItem = dynamic_cast<GraphicsComponentItem*>(items[i]);
            if (gItem->getComponent()->getProperty("id").toString() == name)
                return gItem;
        }

    }
    return 0;
}


void SheetView::mouseMoveEvent(QMouseEvent *event)
{
    if (event->buttons() & Qt::LeftButton)
    {
        scene()->setSceneRect(sceneRect().united(scene()->itemsBoundingRect()));
        setSceneRect(scene()->sceneRect());
        if (rectSelection->isVisible())
        {
            QTransform tr = transform();
            QPointF endPos = mapToScene(tr.map(event->pos()));

            QPointF lowerLeft( MIN(rectSelection->x(),endPos.x()), MIN(rectSelection->y(),endPos.y()) );
            QPointF upperRight(MAX(rectSelection->x(),endPos.x()), MAX(rectSelection->y(),endPos.y()) );

            rectSelection->setRect(QRectF(lowerLeft - rectSelection->pos(), upperRight - rectSelection->pos()));
        }
        //event->setAccepted(true);
        //return;
    }
    QGraphicsView::mouseMoveEvent(event);
}


void SheetView::mousePressEvent(QMouseEvent *event)
{

    if (event->buttons() & Qt::LeftButton)
    {
        QTransform tr = transform();

        QPointF posStart = mapToScene(tr.map(event->pos()));

        QList<QGraphicsItem*> items = scene()->items(posStart);


        if (!items.count())
        {
            rectSelection->setVisible(true);
            rectSelection->setRect(0,0,2,2);
            rectSelection->setPos(posStart);
        }
        //event->setAccepted(true);
        //return;
    }


    QGraphicsView::mousePressEvent(event);
}


void SheetView::mouseReleaseEvent(QMouseEvent *event)
{
    if (rectSelection->isVisible())
    {
       QList<QGraphicsItem*> items = scene()->items(rectSelection->mapToScene(rectSelection->rect()));
        for (int i = 0; i< items.count(); i++)
        {
            if (items[i] != rectSelection)
                items[i]->setSelected(true);
        }
    }
    QGraphicsView::mouseReleaseEvent(event);
    rectSelection->setVisible(false);

    actionCopy->setEnabled(scene()->selectedItems().count());
    actionDelete->setEnabled(scene()->selectedItems().count());
}

void SheetView::contextMenuEvent(QContextMenuEvent *event)
{
    QTransform tr = transform();
    QPointF posStart = mapToScene(tr.map(event->pos()));
    QList<QGraphicsItem*> items = scene()->items(posStart);


    if (items.count() != 0)
    {
        QGraphicsView::contextMenuEvent(event);
    }
    else
    {
        QMenu menu;
        menu.addActions(this->actions());
        menu.exec(event->globalPos());
    }
}


void SheetView::deleteSelection()
{
    // deletion is not a simple task :
    // it should be in the accurate order :
    // - first process components, then regenerate list of selectedItems;
    // - then ports and again regeneration
    // - finally invocations as well as links in any order
    QList<QGraphicsItem*> itemList = scene()->selectedItems();
    itemList = scene()->selectedItems();
    QString name;
    int i;
    for (i=0; i < itemList.count();i++)
    {
        if (itemList[i]->type() == GraphicsComponentItem::Type)
        {
            GraphicsComponentItem* item = dynamic_cast<GraphicsComponentItem*>(itemList[i]);
            name = item->getComponent()->getProperty("id").toString();
            componentList.removeAll(name);
            item->markForDeletion();
            delete item;
            // each time an item is removed from list, we must regenerate list of selected items.
            itemList = scene()->selectedItems();
            i=-1;
        }
    }
    sheet.setProperty("component-list",componentList.join(";"));


    itemList = scene()->selectedItems();
    for (i=0; i < itemList.count();i++)
    {
        if (itemList[i]->type() == GraphicsPortItem::Type)
        {
            GraphicsPortItem* item = dynamic_cast<GraphicsPortItem*>(itemList[i]);
            item->markForDeletion();
            delete item;
            // each time an item is removed from list, we must regenerate list of selected items.
            itemList = scene()->selectedItems();
            i=-1;

        }

    }

    itemList = scene()->selectedItems();
    for (i=0; i < itemList.count(); i++)
    {
        if (itemList[i]->type() == GraphicsInvocationItem::Type)
        {
            GraphicsInvocationItem* item = dynamic_cast<GraphicsInvocationItem*>(itemList[i]);
            item->markForDeletion();
            delete item;
            // each time an item is removed from list, we must regenerate list of selected items.
            itemList = scene()->selectedItems();
            i=-1;

        }
        else
        {
            if (itemList[i]->type() == GraphicsLinkItem::Type )
            {
                GraphicsLinkItem* item = dynamic_cast<GraphicsLinkItem*>(itemList[i]);
                item->markForDeletion();
                delete item;
                // each time an item is removed from list, we must regenerate list of selected items.
                itemList = scene()->selectedItems();
                i=-1;
            }
        }

    }
    actionCopy->setEnabled(scene()->selectedItems().count());
    actionDelete->setEnabled(scene()->selectedItems().count());
    actionSelectAll->setEnabled(false);
}

void SheetView::clearSelection()
{
    QList<QGraphicsItem*> items = scene()->selectedItems();
    for (int i=0; i< items.count(); i++)
        items[i]->setSelected(false);

    actionCopy->setEnabled(scene()->selectedItems().count());
    actionDelete->setEnabled(scene()->selectedItems().count());

}

void SheetView::selectAll()
{
    QList<QGraphicsItem*> items = scene()->items();
    for (int i=0; i< items.count(); i++)
        items[i]->setSelected(true);

    actionCopy->setEnabled(scene()->selectedItems().count());
    actionDelete->setEnabled(scene()->selectedItems().count());

}




void SheetView::copy()
{
    int i;
    // first we create the needed structures
    ARCSCompositeComponent* acc = new ARCSCompositeComponent();
    ARCSContext* copyContext = new ARCSContext();
    ARCSSheet* copySheet = new ARCSSheet(copyContext);

    // now we should look into selected elements.
    // it will be done using two passes.
    // the first one is to look for components.
    // the second one is to assess invocations and links.

    QList<QGraphicsItem*> selectedItems = scene()->selectedItems();

    for (i=0; i < selectedItems.count(); i++)
    {
        if (selectedItems[i]->type() == GraphicsComponentItem::Type)
        {
            ARCSAbstractComponent* origComponent = dynamic_cast<GraphicsComponentItem*>(selectedItems[i])->getComponent();
            if (! copyContext->addComponent(origComponent->getProperty("id").toString(), origComponent->getType(), origComponent->toString()))
                std::cerr << "problem creating component "  << qPrintable(origComponent->getProperty("id").toString()) << std::endl;
        }
    }
    QStringList copiedComponents = copyContext->getComponentList();
    //! \todo we should also take care of graphics details for our duplication.
    copySheet->setProperty("component-list",copiedComponents.join(";"));

    for (i=0; i <  copiedComponents.count(); i++ )
    {
        /* we remap coordinates */
        QString propertyString = sheet.getProperty(copiedComponents[i]+"-pos");
        QPointF pos = QPointF(propertyString.section(",",0,0).toFloat(),
                                                  propertyString.section(",",1,1).toFloat()) ;
        pos -= mapToScene(0,0);
        copySheet->setProperty(copiedComponents[i]+"-pos",QString::number(pos.x())+","+QString::number(pos.y()));
        /* end of remapping */
        copySheet->setProperty(copiedComponents[i]+"-slots",sheet.getProperty(copiedComponents[i]+"-slots"));
        copySheet->setProperty(copiedComponents[i]+"-signals",sheet.getProperty(copiedComponents[i]+"-signals"));
    }

    //! \todo we should also take care of constants.
    // Once the components are selected, we can look for invocations and links.
    for (i=0; i< selectedItems.count(); i++)
    {
        if (selectedItems[i]->type() == GraphicsLinkItem::Type)
        {
            ARCSConnection* origLink = dynamic_cast<GraphicsLinkItem*>(selectedItems[i])->getConnection();
            QString origSource = origLink->getSourceName();
            QString origDestination = origLink->getDestinationName();
            if (copiedComponents.contains(origSource) && copiedComponents.contains(origDestination))
            {
                ARCSConnection& copyLink = copySheet->addConnection(origSource,origLink->getSignalName(),
                                                                    origDestination,origLink->getSlotName(),
                                                                    origLink->isQueued());
                copyLink.setCoordinates(origLink->getCoordinates());
                std::cout << "Connection append" << std::endl;
            }
        }
        if (selectedItems[i]->type() == GraphicsInvocationItem::Type)
        {
            ARCSInit& origInvocation = dynamic_cast<GraphicsInvocationItem*>(selectedItems[i])->getInvocation();
            GraphicsInvocationItem::InvocationType origInvocationType = dynamic_cast<GraphicsInvocationItem*>(selectedItems[i])->getInvocationType();
            QString origDestination = origInvocation.getDestinationName();
            if (copiedComponents.contains(origDestination))
            {
                switch(origInvocationType)
                {
                case GraphicsInvocationItem::Pre:
                    copySheet->addPreconnect(origDestination, origInvocation.getSlotName(),origInvocation.getValueType(), origInvocation.getValueString());
                    break;
                case GraphicsInvocationItem::Post:
                    copySheet->addPostconnect(origDestination, origInvocation.getSlotName(),origInvocation.getValueType(), origInvocation.getValueString());
                    break;
                case GraphicsInvocationItem::Cleanup:
                    copySheet->addCleanup(origDestination, origInvocation.getSlotName(),origInvocation.getValueType(), origInvocation.getValueString());
                    break;
                }
                std::cout << "invocation append" << std::endl;
             }
        }
    }

    acc->setSheet(*copySheet);
    // once all is set, we prepare clipBoard data
    QClipboard *clipboard = QApplication::clipboard();
    QString copyText = acc->toString();
    clipboard->clear();
    clipboard->setText(copyText);

    // last thing, we do some cleanup
    delete acc;
    delete copySheet;
    delete copyContext;
}



void SheetView::paste()
{
    int i,j/*,k*/;
    // first, we should recover data
    QClipboard *clipboard = QApplication::clipboard();
    QString dataString = clipboard->text();
    QPointF offsetPos = mapToScene(0,0);//sceneRect().bottomLeft();

    QRectF rect = sceneRect();
    std::cout << "offsetPos: " << offsetPos.x()  << ", " << offsetPos.y() << std::endl;
    std::cout << "sceneRect: " << rect.x() << ", " << rect.y() << ": " << rect.width() << "x" << rect.height() << std::endl;

    // now we should create a composite component standing on its feets !
    ARCSCompositeComponent* acc = new ARCSCompositeComponent();
    if (!acc->parseString(dataString))
        return ;

    // acc now contains a lot of things.
    // We will have to fuse them with the current sheet as well as reactualize the context !
    ARCSSheet* incomingSheet = acc->getSheet();
    ARCSContext* incomingContext = incomingSheet->getContext();//acc->getContext();

    QStringList actualComponents = sheet.getContext()->getComponentList();
    QStringList incomingComponents = incomingContext->getComponentList();
    QMap<QString,QString> renamedComponents;

    // first pass: we have some components we will have to rename and add them to context
    for (i=0; i < incomingComponents.count(); i++)
    {
        QString candidateComponent = incomingComponents[i];
        ARCSAbstractComponent* aac = incomingContext->getComponent(candidateComponent);
        while (actualComponents.contains(candidateComponent) || renamedComponents.contains(candidateComponent))
        {
            // let's rename the component
            candidateComponent += "_";
        }
        if (actualComponents.contains(incomingComponents[i]))
            renamedComponents[incomingComponents[i]] = candidateComponent;

        std::cout << "Adding one component" << std::endl;
        sheet.getContext()->addComponent(candidateComponent, aac->getType(), aac->toString());

        // let's look for properties inside this.
        QString propertyString;
        if (! (propertyString =incomingSheet->getProperty(incomingComponents[i] + "-slots")).isEmpty())
            sheet.setProperty(candidateComponent+ "-slots",propertyString);
        if (! (propertyString =incomingSheet->getProperty(incomingComponents[i] + "-signals")).isEmpty())
            sheet.setProperty(candidateComponent+"-signals",propertyString);
        if (! (propertyString =incomingSheet->getProperty(incomingComponents[i] + "-pos")).isEmpty())
        {
            QPointF pos = QPointF(propertyString.section(",",0,0).toFloat(),
                                  propertyString.section(",",1,1).toFloat()) + offsetPos ;
            sheet.setProperty(candidateComponent+"-pos",
                              QString::number(pos.x())+","+QString::number(pos.y()));
        }
    }

    // then we also look for the connections and invocations
    QStringList sources;
    QStringList destinations;
    QStringList signalList;
    QStringList slotList;
    QStringList invocationTypes;
    QStringList invocationValues;
    QString source, destination;
    incomingSheet->getConnections(sources,signalList,destinations,slotList);
    for (i=0; i< sources.count(); i++)
    {
        ARCSConnection& c = incomingSheet->getConnection(sources[i],signalList[i],destinations[i],slotList[i]);
        source = (renamedComponents.contains(sources[i]))?renamedComponents[sources[i]]:sources[i];
        destination = (renamedComponents.contains(destinations[i]))?renamedComponents[destinations[i]]:destinations[i];
        ARCSConnection& cbis = sheet.addConnection(source, signalList[i],destination, slotList[i], c.isQueued());
        QList<QPointF> coords = c.getCoordinates();
        for (j=0; j< 4 ; j++)
        {
            coords[i] += offsetPos;
        }
        cbis.setCoordinates(c.getCoordinates());

    }
    incomingSheet->getPreconnects(destinations,slotList,invocationTypes,invocationValues);
    for (i=0; i< destinations.count();i++)
    {
        destination = (renamedComponents.contains(destinations[i]))?renamedComponents[destinations[i]]:destinations[i];
        sheet.addPreconnect(destination, slotList[i], invocationTypes[i], invocationValues[i]);
    }
    incomingSheet->getPostconnects(destinations,slotList,invocationTypes,invocationValues);
    for (i=0; i< destinations.count();i++)
    {
        destination = (renamedComponents.contains(destinations[i]))?renamedComponents[destinations[i]]:destinations[i];
        sheet.addPostconnect(destination, slotList[i], invocationTypes[i], invocationValues[i]);
    }
    incomingSheet->getCleanups(destinations,slotList,invocationTypes,invocationValues);
    for (i=0; i< destinations.count();i++)
    {
        destination = (renamedComponents.contains(destinations[i]))?renamedComponents[destinations[i]]:destinations[i];
        sheet.addCleanup(destination, slotList[i], invocationTypes[i], invocationValues[i]);
    }

    /* right now we clear the scene and make a new layout.
      * selection is cleared and only new components are selected
      * it could help moving the new components in a more interesting location.
      */
    clearScene();
    layoutSheet();

    scene()->clearSelection();
    for(i=0; i < incomingComponents.count(); i++)
    {
        if (renamedComponents.contains(incomingComponents[i]))
            findComponentByName(renamedComponents[incomingComponents[i]])->setSelected(true);
        else
            findComponentByName(incomingComponents[i])->setSelected(true);
    }

    emit requestApplicationUpdate();

    actionCopy->setEnabled(scene()->selectedItems().count());
    actionDelete->setEnabled(scene()->selectedItems().count());
    actionSelectAll->setEnabled(true);
}

void SheetView::orderInvocations()
{


}


void SheetView::orderConnections()
{
    OrderConnectionsDialog* ocd= new OrderConnectionsDialog(&sheet,this);
    ocd->exec();

}
