/*
  name: tools/libdump/main.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/

#include <iostream>
#include <arcs/arcsfactory.h>


/*! \page arcslibdump arcslibdump manpage
 *<H2>NAME</H2>
 *<B>arcslibdump</B>
 *helper tool to dump component librarie contents.
 *<H2>SYNOPSIS</H2>
 *<B>arcslibdump</B>
 *[LIBRARY]...
 *<H2>DESCRIPTION</H2>
 *<B>arcslibdump</B>
 *dumps the contents of a list of component library names passed as a parameter.
 *It can be abridged library names (for example,
 *<I>libsample.so </I>
 *can also be called
 *<I>sample</I>
 *). For each library,
 *<B>arcslibdump</B>
 *will load the library, load families, data types and components.
 *Each component will be instantiated, their signal and slot lists will be
 *retrieved and they will be destroyed. The library will also be unloaded so
 *you may also check behaviours at instantiation and destruction.
 *<H2>EXAMPLES</H2>
 *<DL COMPACT>
 *<DT>
 *<B>arcslibdump sample</B>
 *<DD>
 *<DD>dumps the contents of the
 *<I>libsample.so </I>
 *component library.
 *<DT>
 *<B>arcslibdump sample libcoin.so</B>
 *<DD>
 *<DD>dumps the contents of
 *<I>libsample.so</I>
 *first, then the contents of
 *<I>libcoin.so.</I>
 *</DL>
 *<H2>NOTE</H2>
 *The output of arcslibdump is quite verbose so it may be a good idea to
 *pipe it with tools like
 *<I>more </I>
 *or
 *<I>less.</I>
 *<H2>AUTHOR</H2>
 *Jean-Yves Didier
 */

void usage(char* name)
{
    std::cout << "Usage: " << name << " libraries..." << std::endl;
    std::cout << "This tool is intended to dump library contents. " << std::endl;
    std::cout << "It is designed to debug complex component libraries using component families" << std::endl;
}


void dumpLibrary(char* name)
{
    ARCSFactory* factory = ARCSFactory::getInstance();
    int i,j;

    std::cout << "Library " << name <<  std::endl<< "{" << std::endl;
    factory->loadLibrary(name);

    QStringList familyList = factory->getFamilyNames();
    if (familyList.count())
    {
        familyList.sort();
        std::cout << "\tFamilies" << std::endl << "\t{" << std::endl;
        for(i=0; i<familyList.count(); i++)
        {
            if (factory->getFamilyLibrary(familyList[i]) == name
                    && !familyList[i].startsWith(QString(name)+":"))
                std::cout << "\t\t" << qPrintable(familyList[i]) << ";" << std::endl;
        }
        std::cout << "\t}" << std::endl;
    }

    QStringList typeList = factory->getTypeNames();
    if (typeList.count())
    {
        typeList.sort();
        std::cout << "\tTypes" << std::endl << "\t{" << std::endl;
        for(i=0; i<typeList.count(); i++)
        {
            if (factory->getTypeLibrary(typeList[i]) == name
                    && !typeList[i].startsWith(QString(name)+":"))
                std::cout << "\t\t" << qPrintable(typeList[i]) << ";" << std::endl;
        }
        std::cout << "\t}" << std::endl;
    }

    QStringList componentList = factory->getLibraryComponents(name);
    if (componentList.count())
    {
        componentList.sort();
        std::cout << "\tComponents" << std::endl << "\t{" << std::endl;

        for(i=0; i<componentList.count(); i++)
        {
            std::cout << "\t\t" << qPrintable(componentList[i]) << std::endl;

            ARCSAbstractComponent* aac = factory->createComponent(componentList[i]);
            std::cout << "\t\t{" << std::endl;

            QStringList slotList = aac->getSlots();
            if (slotList.count())
            {
                std::cout << "\t\t\tSlots {"  << std::endl;
                for (j=0; j < slotList.count(); j++)
                    std::cout << "\t\t\t\t" << qPrintable(slotList[j]) << ";" << std::endl;
                std::cout << "\t\t\t}" << std::endl;
            }

            QStringList signalList = aac->getSignals();
            if (signalList.count())
            {
                std::cout << "\t\t\tSignals {"  << std::endl;
                for (j=0; j < signalList.count(); j++)
                    std::cout << "\t\t\t\t" << qPrintable(signalList[j]) << ";" << std::endl;
                std::cout << "\t\t\t}" << std::endl;
            }
            factory->destroyComponent(aac);
            std::cout << "\t\t}" << std::endl;
        }
        std::cout << "\t}" << std::endl;
    }

    std::cout << "}" << std::endl;
    factory->unLoadLibrary(name);
}





int main(int argc, char* argv[])
{
    if (argc <= 1)
    {
        usage(argv[0]);
        return 0;
    }

    for (int i=1; i<argc; i++)
        dumpLibrary(argv[i]);

    return 0;
}
