/*
  name: tools/libmaker/main.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "arcslibmakerparser.h"
#include <QCoreApplication>
#include <QFileInfo>
#include <QDir>
#include <iostream>
#include <fstream>

/*! \page arcslibmaker arcslibmaker manpage
 *<H2>NAME</H2>
 *<B>arcslibmaker </B>
 *helper tool to build component libraries for ARCS.
 *<H2>SYNOPSIS</H2>
 *<B>arcslibmaker </B>
 *[ALXFILE]
 *<H2>DESCRIPTION</H2>
 *<B>arcslibmaker</B>
 *runs in two different modes.
 *<DL COMPACT>
 *<DT>
 *<DD>The first one occurs when
 *<B>arcslibmaker </B>
 *is called without any parameter. It generates a project file for Qt
 *configured with all ARCS requirements.
 *<DT>
 *<DD>The second one occurs when
 *<B>arcslibmaker </B>
 *is called with a parameter file (extension .alx). Actually, this is
 *an XML description of the component library. In this mode, a cpp wrapper
 *is produced that will be compiled with the library.
 *</DL>
 *<H2>EXAMPLES</H2>
 *<DL COMPACT>
 *<DT>
 *<B>arcslibmaker</B>
 *<DD>
 *<DD>produces a project file (.pro) for Qt
 *<DT>
 *<B>arcslibmaker libsample.alx</B>
 *<DD>
 *<DD>produces a cpp wrapper named (alm_libsample.cpp)
 *</DL>
 *<H2>AUTHOR</H2>
 *Jean-Yves Didier
 */


bool useWrapper = false;

QString header_preamble = "#include<arcs/arcslibtoolkit.h>\n#include<QMetaType>\n";
QString header_line = "#include<@>\n" ;

QString type_struct_line = "Q_DECLARE_METATYPE(@)\n";

QString register_preamble = 
"extern \"C\" DLL_EXPORT void arcsLibraryRegister(ARCSComponentMap* cmap," \
" ARCSFamilyMap* fmap, ARCSTypeMap* tmap)\n{\n";
QString unregister_preamble = 
"extern \"C\" DLL_EXPORT void arcsLibraryUnregister(ARCSComponentMap* cmap," \
"ARCSFamilyMap* fmap, ARCSTypeMap* tmap)\n{\n";

QString register_component_line = "\tcmap->insert(\"@\",new ARCSNativeComponentTemplate<@>());\n";
QString register_family_line = "\tfmap->insert(\"@\",new @());\n"; //ARCSFamilyFactoryTemplate<@>());\n";
QString register_variant_line ="\tqRegisterMetaType<@>(\"@\");\n" ;
QString register_type_line = "\tatf = new @();\n" \
                             "\ttmap->insert(atf->getTypeName(), atf);\n";

QString register_type_linebis = "\ttmap->insert(\"@\", new ARCSTypeFactoryTemplate_@());\n" \
"\tqRegisterMetaType<@>(\"@\");\n";
//

QString unregister_component_line = "\tdelete cmap->take(\"@\");\n";
QString unregister_family_line = "\tdelete fmap->take(\"@\");\n";
QString unregister_type_line = "\tdelete tmap->take(\"@\");\n" \
"\tQMetaType::unregisterType(\"@\");\n";
// 

QString qmake_append = 
"\n\nALXFILE = @\nOTHER_FILES += @\narcslibrary.output = alm_${QMAKE_FILE_BASE}.cpp\n" \
"arcslibrary.input = ALXFILE\narcslibrary.commands = arcslibmaker ${QMAKE_FILE_NAME}\n" \
"arcslibrary.variable_out = SOURCES\n" \
"QMAKE_EXTRA_COMPILERS += arcslibrary\n";

QString qmake_preamble = 
     "win32-msvc* {\n TEMPLATE = vclib\n } else {\nTEMPLATE = lib\n}\n" \
     "INCLUDEPATH += $$(ARCSDIR)/include\n" \
     "LIBS += -L$$(ARCSDIR)/lib -larcs\n"\
     "CONFIG += dll\n"\
     "QT = core\n\n" ;

QString qmake_header = "HEADERS += @\n";

QString qmake_source = "SOURCES += @\n";


//#include<QApplication>

/*! 
 * \mainpage
 *
 * Structure gnrale du xml :
 * <ul>
 * <li>library</li>
 * <ul>
 * <li>headers</li>
 * <ul>
 * <li>header</li>
 * </ul>
 * <li>components</li>
 * <ul>
 * <li>component</li>
 * </ul>
 * <li>types</li>
 * <ul>
 * <li>type</li>
 * </ul>
 * <li>families</li>
 * <ul>
 * <li>family</li>
 * </ul>
 * </ul>
 * </ul>
 */ 



QString replace(QString src, QString insert)
{
     QString res = src;
     return src.replace("@", insert).replace("*(", "Star("); //.replace("::","DotDot");
}




QString cppFileBase(char* fn)
{
     QFileInfo fi(fn);

     return fi.canonicalPath() +QDir::separator() + "alm_" + fi.baseName() + ".cpp";
}


void writeHeaders(std::ofstream & file, QStringList hl)
{
     //file << qPrintable(header_preamble);

     for(  int i=0; i < hl.count() ; i++)
	  file << qPrintable(replace(header_line, hl[i]));
     file << std::endl;
}


void writeTypeList(std::ofstream &file, QStringList tl)
{
     for (int i=0; i < tl.count() ; i++)
	  file << qPrintable(replace(type_struct_line,tl[i]));
     file << std::endl;
}



void writeRegisterWrapper(std::ofstream &file, QStringList cl, QStringList fl, QList<ARCSTypeWrapper> tl)
{
     int i;
     file << "extern \"C\" DLL_EXPORT void arcsLibraryRegister(ARCSComponentMap* " ;
     if (cl.count())
          file << "cmap";
     file << ", ARCSFamilyMap* ";
     if (fl.count())
          file << "fmap";
     file << ", ARCSTypeMap* ";
     if (tl.count())
          file << "tmap";
     file << ")\n{\n";


               //qPrintable(register_preamble) ;

     for (i=0; i < cl.count() ; i++)
	  file << qPrintable(replace(register_component_line,cl[i])) ;
     file << std::endl;
     

     for (i=0; i < fl.count() ; i++)
	  file << qPrintable(replace(register_family_line,fl[i]));
     file << std::endl;

     for (i=0; i < tl.count() ; i++)
     {
          ARCSTypeWrapper atw =  tl[i];
          if (atw.getWrapper().isEmpty())
               file << qPrintable(replace(register_type_linebis,atw.getName()));
          else
          {
              if (!useWrapper)
                  file << "\tARCSTypeFactory* atf;" ;
              useWrapper = true;
               file << qPrintable(replace(register_type_line, atw.getWrapper()))
                         << qPrintable(replace(register_variant_line, atw.getName()));
          }
     }

     file << "}" << std::endl << std::endl;
}


void writeUnRegisterWrapper(std::ofstream &file,QStringList cl, QStringList fl, QList<ARCSTypeWrapper> tl)
{
     int i;
     file << "extern \"C\" DLL_EXPORT void arcsLibraryUnregister(ARCSComponentMap* " ;
     if (cl.count())
          file << "cmap";
     file << ", ARCSFamilyMap* ";
     if (fl.count())
          file << "fmap";
     file << ", ARCSTypeMap* ";
     if (tl.count())
          file << "tmap";
     file << ")\n{\n";

//     file << qPrintable(unregister_preamble) ;
     
     
     for (i=0; i < cl.count() ; i++)
	  file << qPrintable(replace(unregister_component_line,cl[i])) ;
     file << std::endl;
     

     for (i=0; i < fl.count() ; i++)
	  file << qPrintable(replace(unregister_family_line,fl[i]));
     file << std::endl;

     for (i=0; i < tl.count() ; i++)
     {
          ARCSTypeWrapper atw= tl[i];
         //if (atw.getWrapper().isEmpty())
          //{
          file << qPrintable(replace(unregister_type_line,atw.getName()));
     }

     file << "}" << std::endl << std::endl;
}


void appendProjectFile(std::ofstream & of)
{
     QStringList alxFilter;
     alxFilter << "*.alx";

     QDir dir = QDir::current() ; // le rep courant
     dir.refresh();
     QStringList alxFiles = dir.entryList(alxFilter, QDir::Files | QDir::Readable | QDir::Writable );
     
     if (alxFiles.count() <= 0 )
	  return ;

     of << qPrintable(replace(qmake_append, alxFiles[0])) << std::endl;

}



void projectMode()
{
     int i;
     QStringList proFilter;
     proFilter << "*.pro";

     QDir dir = QDir::current() ; // le rep courant
     QStringList proFiles = dir.entryList(proFilter, QDir::Files | QDir::Readable | QDir::Writable );

     std::ofstream of;

     if (proFiles.count() > 0)
     {
	  of.open(qPrintable(proFiles[0]),std::ios_base::app);
	  if (!of.is_open())
	       return ;
     }
     else
     {
	  // 1 : rcuprer le nom de projet
	  QString proName = QDir::current().dirName() + ".pro" ;
	  of.open(qPrintable(proName));
	  if (!of.is_open())
	       return ; 

	  // 2 : crer le fichier
	  QString  path = QDir(QCoreApplication::applicationDirPath() + QDir::separator() + "..").canonicalPath() ;
	  of << qPrintable(replace(qmake_preamble, path));


	  // 3 : ajouter les infos sur les .h et les .cpp
	  QStringList headerFilter;
	  headerFilter << "*.h";
	  QStringList headerFiles = dir.entryList(headerFilter, QDir::Files);
	  for (i = 0; i < headerFiles.count() ; i++)
	       of << qPrintable(replace(qmake_header, headerFiles[i]));

	  QStringList sourceFilter;
	  sourceFilter << "*.cpp" << "*.c";
	  QStringList sourceFiles = dir.entryList(sourceFilter, QDir::Files);
	  for (i = 0; i < sourceFiles.count() ; i++)
	       of << qPrintable(replace(qmake_source, sourceFiles[i]));

     }

     // tester ici si un fichier alx existe, sinon le crer avec un squelette.
     QStringList alxFilter(QString("*.alx"));
     QStringList alxFiles = dir.entryList(alxFilter, QDir::Files | QDir::Readable | QDir::Writable );
     if( alxFiles.count() <= 0)
     {
          QString alxTarget = "lib"+QDir::current().dirName() + ".alx" ;
          if (QFile::copy(":/alx_skeleton.xml",alxTarget))
               QFile::setPermissions(alxTarget,QFile::permissions(alxTarget) | QFile::WriteOwner | QFile::WriteUser);
     }


     appendProjectFile(of);
     of.close();
}


void usage(char* prg)
{
     std::cout << prg << " [--help] [file]" << std::endl;
     
     std::cout << prg << " has two modes, one for generating ARCS library wrappers," 
	       << std::endl << " the second for adding ARCS options to Qt project files." 
	       << std::endl << std::endl 
	       << "The first mode needs an xml file describing the library contents." 
	       << std::endl;
}

int main(int argc, char* argv[])
{
     QCoreApplication app(argc, argv);

     if (argc > 1)
     {
	  if (QString(argv[1]) == "--help")
	  {
	       usage(argv[0]);
	       return 0;
	  }
	  else
	  {
	       ARCSLibMakerParser almp ;
	       almp.setFileName(argv[1]);
	       if (almp.parseFile())
	       {
		    //std::cout << qPrintable(cppFileBase(argv[1])) << std::endl;
		    std::ofstream of( qPrintable(cppFileBase(argv[1])) );
		    if (! of.is_open())
			 return 2;
		  of << qPrintable( header_preamble ) ;	    
		  //writeTypeList(of, almp.getTypes());
		    writeHeaders(of, almp.getHeaders());
		    //writeTypeList(of, almp.getTypes());
		    writeRegisterWrapper(of, almp.getComponents(), almp.getFamilies(), almp.getTypes());
		    writeUnRegisterWrapper(of, almp.getComponents(), almp.getFamilies(), almp.getTypes());
		    of.close();
	       }
	       else
		    return 1;
	  }
     }
     else
     {
	  projectMode();
     }

     return 0;
}
