/*
  name: tools/pkg/arcsdownloader.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "arcsdownloader.h"
#include <QNetworkReply>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QFileInfo>
#include <iostream>
#include <QAuthenticator>
#include <QSslCertificate>
#include <QEventLoop>
#include <getpasswd.h>

#include <string>

ARCSDownloader::ARCSDownloader(QObject *parent) :
    QObject(parent)
{
    manager = new QNetworkAccessManager(this);
    eventLoop = new QEventLoop(this);
    connect(manager,SIGNAL(proxyAuthenticationRequired(QNetworkProxy,QAuthenticator*)),
            this,SLOT(authenticateToProxy(QNetworkProxy,QAuthenticator*)));
    connect(manager,SIGNAL(sslErrors(QNetworkReply*,QList<QSslError>)),
            this,SLOT(sslErrors(QNetworkReply*,QList<QSslError>)));
    connect(manager,SIGNAL(authenticationRequired(QNetworkReply*,QAuthenticator*)),
            this,SLOT(authenticateToSite(QNetworkReply*,QAuthenticator*)));
    connect(manager,SIGNAL(finished(QNetworkReply*)),this,SLOT(replyFinished(QNetworkReply*)));

}


void ARCSDownloader::replyFinished(QNetworkReply *reply)
{
    if (!reply)
        return ;
    reply->deleteLater();
    QString urlString = reply->request().url().toString();

    if (reply->error() != QNetworkReply::NoError)
    {
        std::cerr << "Downloader has encountered an error: "
                  << qPrintable(reply->errorString()) << std::endl;
        std::cerr << "on url: " << qPrintable(urlString) << std::endl;
        // should remove wrong file from downloads
        eventLoop->quit();
        return;
    }

    QFile file(filename);
    if (!file.open(QIODevice::WriteOnly))
    {
        std::cerr << "Failed to write down file: " << qPrintable(filename) << std::endl;
        eventLoop->quit();
        return;
    }

    //std::cout << "Downloaded !" << std::endl; //<< qPrintable(filename) << std::endl;
    file.write(reply->readAll());
    eventLoop->quit();
}

void ARCSDownloader::authenticateToSite(QNetworkReply * reply, QAuthenticator * auth)
{
    std::cout << "The site you try to connect requires you to authenticate yourself." << std::endl;
    authenticate(auth);
}

void ARCSDownloader::authenticateToProxy(const QNetworkProxy &proxy, QAuthenticator *auth)
{
    std::cout << "Your network proxy requires you to authenticate yourself." << std::endl;
    authenticate(auth);
}

void ARCSDownloader::sslErrors(QNetworkReply *reply, const QList<QSslError> &errors)
{
    std::cerr << "Your ssl connection has encountered the following errors " << std::endl;
    for (int i=0; i < errors.count(); i++)
    {
        std::cerr << "  " << qPrintable(errors[i].errorString()) << std::endl;
        QSslCertificate cert = errors[i].certificate();
        std::cerr << "  Certificate issued by " << qPrintable(cert.subjectInfo(QSslCertificate::CommonName))
                  << ", " << qPrintable(cert.subjectInfo(QSslCertificate::Organization))
                  << ", " << qPrintable(cert.subjectInfo(QSslCertificate::OrganizationalUnitName))
                  << ", " << qPrintable(cert.subjectInfo(QSslCertificate::LocalityName))
                  << ", " << qPrintable(cert.subjectInfo(QSslCertificate::CountryName))
                  << std::endl;

    }
    std::cerr << "Do you wish to continue (y/N) ?";
    std::string r;
    std::cin >> r;
    if (r.at(0) == 'y')
    {
        reply->ignoreSslErrors();//errors);
        cachedSslErrors = errors;
    }
    else
    {
        std::cerr << "Ssl transaction has been dropped" << std::endl;
        eventLoop->quit();
    }
}

void ARCSDownloader::setProgress(qint64 received, qint64 total)
{
    qint64 frac=received*40/total;
    qint64 percent = received*100/total;

    std::cout << "\rDownloading " << qPrintable(filename) << "\t[";
    for (int i=0; i<40; i++)
    {
        if (i < frac)
            std::cout << '=';
        else
        {
            if (i == frac)
                std::cout << "+";
            else
                std::cout << "-";
        }
    }
    std::cout << "] " <<  percent << "%\t" << std::flush;
    if (frac==40) std::cout << std::endl;
}

void ARCSDownloader::authenticate(QAuthenticator *auth)
{
    std::cout << "Username: " ;
    std::string uname;
    std::cin >> uname;
    std::string pwd;
    pwd = getpasswd("Password: ");
    auth->setUser(uname.data());
    auth->setPassword(pwd.data());
}

void ARCSDownloader::download(QString s)
{
    QNetworkReply* reply = manager->get(QNetworkRequest(s));
    reply->ignoreSslErrors(cachedSslErrors);
    filename = QFileInfo(reply->request().url().path()).fileName();
    std::cout << "Trying to download " << qPrintable(filename) << std::endl;
    connect(reply,SIGNAL(downloadProgress(qint64,qint64)),this, SLOT(setProgress(qint64,qint64)));
    eventLoop->exec();
}
