/*
  name: include/arcs/arcscontext.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSCONTEXT_H__
#define __ARCSCONTEXT_H__

#include <arcs/arcsfactory.h>
#include <iostream>
//#include <QHash>


/*! \brief This class manages components and constants in a given context.
 *
 * 
 * This class encapsulates a component pool as well as a constant pool.
 * \ingroup lib
 * \author Jean-Yves Didier
 * \date December, the 4th, 2008
 */
class DLL_EXPORT ARCSContext
{
public:
     //! Constructor
     ARCSContext();
     //! Copy constructor
     ARCSContext(const ARCSContext& ctx); //{ std::cout << "###### Attempt to use the copy constructor !" << std::endl;}
     virtual ~ARCSContext();

     //! Resets the component and constant lists 
     void reset();

     //! Adds a component to the list.
     /*! A component is characterized by its name and its class name.
      * \param name id of the component. Should be unique inside a context.
      * \param className class name of the component.
      * \param contents a string representing a component.
      */
     bool addComponent(QString name, QString className, QString contents = QString::null );


     bool renameComponent(QString oldName, QString newName);


     ARCSAbstractComponent* createComponent(QString name, QString className);

     //! Removes a component from the component pool.
     /*! \param name id of the component to remove
      */
     void removeComponent(QString name);

     //! Retrieves a component by its id.
     /*! \param name id of the component.
      */
     ARCSAbstractComponent* getComponent(QString name);
     
     /*! Returns the list of components (their Id's) */
     QStringList getComponentList();
     /*! Returns the list of constants (their Id's) */
     QStringList getConstantList();


     //! Adds a constant to the list.
     /*!\param name id of the constant
      * \param type type of the constant
      * \param representation string representing a constant.
      */
     bool addConstant(QString name, QString type, QString representation);

     //! Modifies a constant of the list
     /*! \param name id of the constant
      * \param representation string representing a constant.
      */
     bool modifyConstant(QString name, QString representation);

     bool modifyConstant(QString name, QString type, QString representation)
     {
          removeConstant(name);
          addConstant(name, type, representation);
          return true;
     }

     //! Removes a constant from the constant pool.
     /*! \param name id of the constant
      */
     void removeConstant(QString name);


     bool isInstanciated() { return instanciated;}

     //! Retrieves a constant by its id.
     /*! \param name id of the constant to retrieve.
      */
     QVariant getConstant(QString name);


     //! Instanciates the whole pool of components inside the context
     bool instanciatePool();


     QVariant createValue(QString type, QString representation);


     /*!  Computes the list of libraries needed by this context  */
     QStringList computeLibraryList();


     //! Serializes constants
     /*! 
      * \param name Constant name
      * \param type Type of the constant
      * \param representation Representation of the constant
      * \return <tt>true</tt> if serialization could be achieved, <tt>false</tt> otherwise
      */
     bool serializeConstant(QString name, QString & type, QString & representation);


     typedef bool (ARCSContext::*AddPoolMethod) (QString, QString, QString);

private:
     QHash<QString, ARCSAbstractComponent*> componentPool;
     QHash<QString, QVariant> constantPool;
     ARCSFactory* factory;
     bool instanciated;
};





#endif //__ARCSCONTEXT_H__
