/*
  name: include/arcs/arcsfactory.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSFACTORY_H__
#define __ARCSFACTORY_H__

#include <arcs/arcslibmanager.h>

#include <arcs/arcsnativefamily.h>

#include <QString>
#include <QMap>

/*! \defgroup lib ARCS Library
 *
 * This module defines the runtime primitives used by ARCS.
 */

/*! \brief A singleton handling all needed factories in ARCS runtime.
 *
 * There are currently 3 groups of factories supported inside ARCS :
 * <ul>
 * <li>Families of component factories;</li>
 * <li>Native component factories;</li>
 * <li>Simple types factories.</li>
 * </ul>
 * 
 * ARCSFactory is implemented as a singleton.
 * \author Jean-Yves Didier
 * \date November, 2008
 * \ingroup lib
 */
class DLL_POINT ARCSFactory
{
     public :
     /*! \brief Returns the instance of the singleton ARCSFactory
      */
     static ARCSFactory* getInstance(); // { return instance; }
     ~ARCSFactory();

     /*! \brief Loads a library given the path to it.
      */
     bool loadLibrary(QString path);
     /*! \brief Unloads a library given the path to it.
      */
     void unLoadLibrary(QString path);

     /*! \brief Returns the names of the currently loaded libraries.
      */
     QStringList getLibraryNames() { return libraries.keys(); }

     /*! \brief Returns the names of the types currently stored. 
      */
     QStringList getTypeNames()    { return types.keys(); }

     /*! \brief Returns the names of the family currently stored.
      */
     QStringList getFamilyNames()  { return families.keys(); }

     /*! \brief Returns the class names of the components currently stored.
      */
     QStringList getComponentNames()  { return components; }
     
     /*! \brief Creates a component given its class name.
      */
     ARCSAbstractComponent* createComponent(QString componentType);
     /*! \brief Destroys a component.
      */
     void destroyComponent(ARCSAbstractComponent* aac);


     /*! \brief Returns the path of the library associated to a certain kind of component
      *
      * \param s type name of the component
      * \returns a library path
      */
     QString getComponentLibrary(QString s);


     /*! \brief Returns the list of components associated to a library.
      *
      * \param s library name or library path
      * \returns a list of component names.
      */
     QStringList getLibraryComponents(QString s); // { return QStringList(); }


     /*! \brief Returns the type of a QVariant
      * \param var a QVariant
      * \returns the type of the QVariant as a string.
      */
     QString getVariantType(QVariant var);


     QString getInternalType(QString s);


     /*! \brief Returns the path of the library associated to a certain kind of type.
      *
      * \param s
      * \return a library path
      */
     QString getTypeLibrary(QString s);

     //! \todo est ce vraiment ncessaire de rendre cette mthode publique ?
     QString getFamilyLibrary(QString s);


     //! Deserialize data from their string representation
     QVariant dataDeserialize(QString type, QString representation);

     //! Serializes data from their QVariant counterpart
     QString dataSerialize(QVariant var);

private:
     void registerInternalFamily(ARCSAbstractFamily* family);
     void registerInternalType(ARCSTypeFactory* type);


     void buildComponentLists();

     ARCSFactory();
     static ARCSFactory* instance;

     // mauvais typage
     //ARCSComponentMap components;
     ARCSFamilyMap families;
     ARCSTypeMap types;

     // mapping  mettre en place : 
     QMap<QString, QString> typeLibraryMap ;
     QMap<QString, QString> nativeComponentLibraryMap ; // fait !!!
     QMap<QString, QString> familyLibraryMap ;
     QMap<QString, QString> variantTypeMap;

     QMap<QString, QString> componentFamilyMap ; // ne pas toucher !!!!
     QStringList components; // ne devrait on pas regnrer ?

     QHash<QString,ARCSLibManager*> libraries;

     ARCSNativeFamily* nativeFamily;

};

#endif //__ARCSFACTORY_H__
