/*
  name: include/arcs/arcslibtoolkit.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSLIBTOOLKIT_H__
#define __ARCSLIBTOOLKIT_H__


/*! \file arcslibtoolkit.h
 *  \author Jean-Yves Didier 
 *  \date October, 27th 2007
 * \ingroup lib
 * 
 * This the header declaring basic classes and functionnalities 
 * to easily create a library designed for our runtime. It stores 
 * a generic mecanism of wrapping that helps to call the constructor of any
 * object inheriting from QObject.
 * One can create it's own library for our runtime, by including this file in
 * the source code of its own library.
 *
 */

#include <QHash>
#include <QMetaObject>

#include <arcs/arcsabstractfamily.h>
#include <arcs/arcsnativecomponent.h>



//! Generic factories for each new native component type.
/*! \author Jean-Yves Didier
  * \date October, 27th 2007
  * \ingroup lib
  */
template<class X> class ARCSNativeComponentTemplate : public ARCSNativeComponent
{
protected:
     virtual ARCSNativeComponent* genuineClone();
     virtual QObject* createObject(QObject* parent);
     virtual void destroyObject(QObject* obj);
     virtual QMetaObject getDescription();
};





//! Generic class describing how type factories should be implemented
/*! All new type made available to the ARCS engine must have such a type factory.
  * Each new factory should derive from this class.
  * If you use the automated process to create your libraries, please notice that
  * your type factory should use ARCSTypeFactoryTemplate.
  *\author Jean-Yves Didier
  * \date October, 27th, 2007
  * \ingroup lib
  */
class ARCSTypeFactory
{
public:
     virtual ~ARCSTypeFactory() {}
     //! Should return the name of the type factory
     virtual QString getTypeName() const =0;
     //! Should return a string representation of data
     virtual QString toString(QVariant v) = 0;
     //! Should create data from their string representation
     virtual QVariant parseString(QString s) = 0;
     //! Tells wether the factory is internal or not (false by default).
     virtual bool isInternal() const { return false;}
};


//! Template class to implement type factories.
/*! This class adds two more methods to implement.
  * Usually, in order to make a new type available to ARCS, one need to derive from this template and reimplement
  * the methods getTypeName(), parse() and serialize().
  * If you use the automated process, the name of your derived class should conform to the
  * following syntax: <tt>ARCSTypeFactoryTemplate_&lt;yourtypeclass&gt;</tt>.
  * As an example here is partly how the type factory for integers is declared inside of ARCS.
  * \code
  class ARCSTypeFactory_int : public ARCSTypeFactoryTemplate<int>
  {
  public:
      virtual QString getTypeName() const { return "int"; }

  protected:
      virtual int parse(QString s) { return s.toInt(); }
      virtual QString serialize(int obj) { return QString::number(obj);}
  };
  \endcode
  * \author Jean-Yves Didier
  * \date October, 27th, 2007
  * \ingroup lib
 */
template<class X> class ARCSTypeFactoryTemplate : public ARCSTypeFactory 
{ 
public: 
     virtual QString getTypeName() const = 0; 
     virtual QString toString(QVariant v);
     virtual QVariant parseString(QString s);
     
protected:
     //! This function should return a X object according to its string representation.
     virtual X parse(QString s) = 0;
     //! This function serializes an X object to a string.
     virtual QString serialize(X obj) = 0;
}; 



#ifndef DLL_EXPORT
#ifdef WIN32
/*! Macro defining keywords for dllexports for MSVC
 *
 * This macro will help export necessary dll entry points in Microsoft Windows environment
 */
#define DLL_EXPORT __declspec(dllexport) 
template class DLL_EXPORT QHash<QString,ARCSNativeComponent*>;
template class DLL_EXPORT QHash<QString,ARCSAbstractFamily*>;
//template class DLL_EXPORT QHash<QString,ARCSFamilyFactory*>;
template class DLL_EXPORT QHash<QString,ARCSTypeFactory*>;
#else
#define DLL_EXPORT
#endif //WIN32
#endif //DLL_EXPORT


/*! Type storing wrapping classes in one end and a key, which is a string representing
 * the name of the object we want to create.
 */
typedef QHash<QString,ARCSNativeComponent*> ARCSComponentMap;

typedef QHash<QString,ARCSAbstractFamily*> ARCSFamilyMap;

typedef QHash<QString,ARCSTypeFactory*> ARCSTypeMap;


/*! Dirty way to implement wrapper since it is in the header file (don't do this at home !), 
 * basically for the sole purpose to only have one file 
 * to include with the user library.
 */
template<class X> 
QObject* ARCSNativeComponentTemplate<X>::createObject(QObject* parent)
{
     return (new X(parent));
}

//! Dirty (Kids, don't do this at home !)
template<class X>
void ARCSNativeComponentTemplate<X>::destroyObject(QObject* obj)
{
     X* xobj = dynamic_cast<X*>(obj);
     if (xobj != NULL)
	  delete xobj;
     else
	  delete obj;
}

template<class X>
QMetaObject ARCSNativeComponentTemplate<X>::getDescription()
{
     setType((X::staticMetaObject).className());
     return X::staticMetaObject;
}

template<class X>
ARCSNativeComponent* ARCSNativeComponentTemplate<X>::genuineClone()
{
     ARCSNativeComponentTemplate<X>* res = new ARCSNativeComponentTemplate<X>();
     return res;
}


template<class X>
QString ARCSTypeFactoryTemplate<X>::toString(QVariant v)
{
     return serialize(v.value<X>());
}

template<class X>
QVariant ARCSTypeFactoryTemplate<X>::parseString(QString s)
{
	QVariant var ;
	var.setValue(parse(s));
     return var;
}


#endif //__ARCSLIBTOOLKIT_H__
