/*
  name: include/arcs/arcslog.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSLOG_H__
#define __ARCSLOG_H__

#include <QList>
#include <QDataStream>
#include <QTextStream>
#include <arcs/arcsdll.h>


#define ARCS_SOURCE __FILE__":"+QString::number(__LINE__)

/*! \brief class that describes a log event
 *
 * \see ARCSLog
 * \author Jean-Yves Didier
 * \date February, the 14th, 2011
 * \ingroup lib
 */
class ARCSLogEvent
{
public:
     //! Level of logging
     enum Level {
          NONE, //!< not a proper level
          INFORMATION, //!< for informational purpose
          WARNING, //!< this may be an error
          ERROR, //!< this is an error but will not cause troubles to the engine
          CRITICAL //!< this is a critical error, it should make the engine stop
     };

     /**
      * @brief Constructor of the event
      * @param src source of the event
      * @param lvl level of the event
      * @param msg message carried out by the event
      */
     ARCSLogEvent(QString src=QString::null, Level lvl=NONE, QString msg=QString::null);

     //! returns the source of the event
     QString getSource() const { return source; }
     //! returns the criticality level of the event
     Level getLevel() const { return level; }
     //! returns the message associated to the event
     QString getMessage()  const{ return message; }

private:
     QString source;
     Level level;
     QString message;
};


QDataStream &operator<< (QDataStream &out, const ARCSLogEvent & event);
QDataStream &operator>> (QDataStream &in, ARCSLogEvent & event);

//! A special QString subclass
class ARCSLogSource :public QString
{
public:
     ARCSLogSource(const char* str, int num) :
         QString(QString(str)+":"+QString::number(num)) {}
};


#ifdef WIN32
#include <windows.h>
#endif


#ifdef ERROR
#undef ERROR
#endif

/*********************************************************************************/
// ARCSColorLog Family
/*********************************************************************************/

/**
 * \brief generic class for coloring log event
 *
 * \author Jean-Yves Didier
 * \date February, the 14th, 2011
 * \ingroup lib
 */

class ARCSColorLog
{
public:
     //!< List of colors
     enum Colors { NORMAL=0, GREEN, BLUE, YELLOW, RED};
     ARCSColorLog(Colors color=NORMAL) { this->color = color;  }
     virtual ~ARCSColorLog() {}
     friend std::ostream& operator<<(std::ostream& os, const ARCSColorLog& cl);
     friend QTextStream& operator<<(QTextStream& ts, const ARCSColorLog& cl);
     //static ARCSColorLog levels[];

protected:
     virtual void log(std::ostream& ) const;
     virtual void log(QTextStream& ) const;
     Colors getColor() const { return color; }

private:
     Colors color;
};

std::ostream& operator<<(std::ostream& os, const ARCSColorLog& cl);
QTextStream& operator<<(QTextStream& ts, const ARCSColorLog& cl);

#ifdef WIN32
//! specialized log colorizer for Windows
class ARCSWindowsColorLog : public ARCSColorLog {
public:
    ARCSWindowsColorLog(Colors color=NORMAL) : ARCSColorLog(color) {}


protected:
    virtual void log(std::ostream& os) const;

private:
    static WORD normalAttribute;
    static WORD getNormalAttribute();
    static WORD attributes[];
};

#else
//! specialized log colorizer for Unix
class ARCSUnixColorLog : public ARCSColorLog {
public:
    ARCSUnixColorLog(Colors color=NORMAL) : ARCSColorLog(color) {}

protected:
    virtual void log(std::ostream& os) const;

private:
    static const char* normalAttribute;
    static const char* attributes[];
};
#endif


//! specialized log colorizer for html output
class ARCSHTMLColorLog : public ARCSColorLog {
public:
    ARCSHTMLColorLog(Colors color=NORMAL) : ARCSColorLog(color) {}

protected:
    virtual void log(std::ostream& os) const;
    virtual void log(QTextStream& ts) const;

private:
    static const char* normalAttribute;
    static const char* attributes[];
};

/******************************************************************************/
// ARCSLog Class
/******************************************************************************/
/**
 * \brief General purpose logging class
 *
 * The ARCSLog class may be used as a logging system.
 * Based, on the singleton pattern, you request it by the getInstance() method.
 * It has several log modes in order to produce colorized output or not.
 *
 * \author Jean-Yves Didier
 * \date February, the 14th, 2011
 * \ingroup lib
 */
class DLL_POINT ARCSLog
{
private:
     ARCSLog();
     static ARCSLog* instance;

     QList<ARCSLogEvent> logEvents;
     ARCSLogEvent::Level display;
     ARCSColorLog* levels[5];
     QTextStream* textStream;

     void refreshMode();

public:
     //! logging modes
     enum LogModes { NONE, CONSOLE, HTML };
     //! returns the only instance of the logging system that is available
     static ARCSLog* getInstance();

     void startCapture() {
         sCapture.clear();
         tsCapture.setString(&sCapture);
         setTextStream(&tsCapture);
     }
     QString endCapture() { setTextStream(); return sCapture; }


     void setTextStream(QTextStream* ts=0) {
         textStream = ts; logMode = HTML; refreshMode(); }
     void setLogMode(LogModes mode) { logMode = mode; refreshMode();  }
     void log(ARCSLogEvent evt);
     static void logInformation(QString source, QString message)
     {
          getInstance()->log(ARCSLogEvent(source,ARCSLogEvent::INFORMATION,message));
     }

     static void logWarning(QString source, QString message)
     {
          getInstance()->log(ARCSLogEvent(source,ARCSLogEvent::WARNING,message));
     }

     static void logError(QString source, QString message)
     {
          getInstance()->log(ARCSLogEvent(source,ARCSLogEvent::ERROR,message));
     }

     static void logCritical(QString source, QString message)
     {
          getInstance()->log(ARCSLogEvent(source,ARCSLogEvent::CRITICAL,message));
     }

     static const char* interp[];
     void setDisplayLevel(ARCSLogEvent::Level lvl) { display = lvl; }

private:
     LogModes logMode;

     QTextStream tsCapture;
     QString sCapture;

};


#endif // __ARCSLOG_H__
