/*
  name: lib/arcsfactory.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcsfactory.h>
#include <arcs/arcsinternalfamily.h>
#include <arcs/arcsinternaltypes.h>
#include <arcs/arcssensorlogger.h>
#include <arcs/arcsgenerallogger.h>
#include <arcs/arcsconstantmodifier.h>
#include <arcs/arcsapplicationobject.h>
#include <QtDebug>
#include <QHashIterator>
#include <arcs/arcslog.h>


QString ARCSAbstractFamily::name() const
{
     return QString("__");
}

bool ARCSAbstractFamily::isInternal() const
{
     return false;
}



ARCSFactory* ARCSFactory::instance = 0;

ARCSFactory::ARCSFactory()
{
     /* First create families */
     nativeFamily = new ARCSNativeFamily();
     families.insert(nativeFamily->name(), nativeFamily);
     registerInternalFamily(new ARCSInternalFamily());
     
     registerInternalType(new ARCSTypeFactory_void());
     registerInternalType(new ARCSTypeFactory_bool());
     registerInternalType(new ARCSTypeFactory_int());
     registerInternalType(new ARCSTypeFactory_short());
     registerInternalType(new ARCSTypeFactory_long());
     registerInternalType(new ARCSTypeFactory_QString());
     registerInternalType(new ARCSTypeFactory_float());
     registerInternalType(new ARCSTypeFactory_double());
     registerInternalType(new ARCSTypeFactory_ARCSConstant());
     qRegisterMetaType<ARCSConstant>("ARCSConstant");
     registerInternalType(new ARCSTypeFactory_ARCSComponent());
     qRegisterMetaType<ARCSComponent>("ARCSComponent");
     registerInternalType(new ARCSTypeFactory_QSize());

     registerInternalType(new ARCSTypeFactory_boolArray());
     qRegisterMetaType<boolArray>("boolArray");
     registerInternalType(new ARCSTypeFactory_intArray());
     qRegisterMetaType<intArray>("intArray");
     registerInternalType(new ARCSTypeFactory_shortArray());
     qRegisterMetaType<shortArray>("shortArray");
     registerInternalType(new ARCSTypeFactory_longArray());
     qRegisterMetaType<longArray>("longArray");
     registerInternalType(new ARCSTypeFactory_floatArray());
     qRegisterMetaType<floatArray>("floatArray");
     registerInternalType(new ARCSTypeFactory_doubleArray());
     qRegisterMetaType<doubleArray>("doubleArray");

     nativeFamily->addFactory("ARCSSensorLogger", new ARCSNativeComponentTemplate<ARCSSensorLogger>());
     nativeFamily->addFactory("ARCSGeneralLogger", new ARCSNativeComponentTemplate<ARCSGeneralLogger>());
     nativeFamily->addFactory("ARCSConstantModifier", new ARCSNativeComponentTemplate<ARCSConstantModifier>());
     nativeFamily->addFactory("ARCSApplicationObject", new ARCSNativeComponentTemplate<ARCSApplicationObject>());

     buildComponentLists();
}

ARCSFactory::~ARCSFactory()
{

}




void ARCSFactory::registerInternalFamily(ARCSAbstractFamily* family)
{
     families.insert(family->name(), family);
}


void ARCSFactory::registerInternalType(ARCSTypeFactory* type)
{
     QVariant var = type->parseString(QString::null);
     QString varName = var.typeName();
     if ( !variantTypeMap.contains(varName))
	  variantTypeMap.insertMulti(varName, type->getTypeName());

     types.insert(type->getTypeName(), type);
}

ARCSFactory* ARCSFactory::getInstance()
{
     if (instance == 0)
	  return (instance = new ARCSFactory());
     else
	  return instance;
}


void ARCSFactory::buildComponentLists()
{
     QHashIterator<QString, ARCSAbstractFamily*> i(families);
     int j;

     components.clear();
     componentFamilyMap.clear();

     while(i.hasNext())
     {
	  i.next();

	  QStringList flist = (i.value())->factoryList();
	  for (j=0;  j < flist.count() ; j++)
	  {
	       if ( components.contains(flist[j]))
              ;//qCritical() << "[Factory] Component " << flist[j] << "already in components.";
	       else
		    components << flist[j];

	       if (componentFamilyMap.contains(flist[j]))
              ;//qCritical() << "[Factory] Component " << flist[j] << "already in componentFamilyMap.";
	       else
		    componentFamilyMap.insert(flist[j], i.value()->name());

	  }

     }

}


bool ARCSFactory::loadLibrary(QString path)
{
     ARCSLibManager* lib = new ARCSLibManager(path);

     if ( ! lib->isLoaded())
     {
         ARCSLog::logError(ARCS_SOURCE,lib->getError());
          delete lib;
          return false;
     }

     if ( libraries.contains(path) )
     {
         ARCSLog::logWarning(ARCS_SOURCE,"library "+ path+ " already loaded.");
          delete lib;
          return true;
     }

     ARCSComponentMap acm = lib->getComponentMap();
     ARCSFamilyMap afm = lib->getFamilyMap();
     ARCSTypeMap atm = lib->getTypeMap();
     
     // parcourir la liste des composants : maj de native
     QHashIterator<QString, ARCSNativeComponent*> i(acm);
     
     while(i.hasNext())
     {
	  i.next();
	  QString namespaced = path + ":" + i.key();
	  
	  nativeFamily->addFactory(namespaced, i.value());
	  nativeFamily->addFactory(i.key(),i.value());

	  if ( nativeComponentLibraryMap.contains(namespaced) )
	       qCritical() << "[Factory] Native component" << namespaced << "already found !";
	  else
	       nativeComponentLibraryMap.insert(namespaced, path);
	  if ( nativeComponentLibraryMap.contains(i.key()) )
	       qCritical() << "[Factory] Native component" << i.key() << "already found !";
	  else
	       nativeComponentLibraryMap.insert(i.key(), path);
     }


     // parcours des familles
     QHashIterator<QString, ARCSAbstractFamily*> j(afm);

     while(j.hasNext())
     {
	  j.next();
	  QString namespaced = path + ":" + j.key();


	  if (families.contains(namespaced))
	       qCritical() << "[Factory] Component family" << namespaced << "already added !";
	  else
	       families.insert(namespaced, j.value());
	  
	  if (families.contains(j.key()))
	       qCritical() << "[Factory] Component family" << j.key() << "already added !";
	  else
	       families.insert(j.key(), j.value());
	  
	  if (familyLibraryMap.contains(namespaced))
	       qCritical() << "[Factory] Component family" << namespaced << "already found !" ;
	  else
	       familyLibraryMap.insert(namespaced, path);
	  
	  if (familyLibraryMap.contains(j.key()))
	       qCritical() << "[Factory] Component family" << j.key() << "already found !" ;
	  else
	       familyLibraryMap.insert(j.key(), path);
     }


     // parcours des types
     QHashIterator<QString, ARCSTypeFactory*> k(atm);
     while(k.hasNext())
     {
	  k.next();
	  QString namespaced = path + ":" + k.key();
	  
	  if (typeLibraryMap.contains(namespaced))
	       qCritical() << "[Factory] Type factory" << namespaced << "already found !";
	  else
	       typeLibraryMap.insert(namespaced, path);

	  if (typeLibraryMap.contains(k.key()))
	       qCritical() << "[Factory] Type factory" << k.key() << "already found !" ;
	  else
	       typeLibraryMap.insert(k.key(), path);
	  
	  if( types.contains(namespaced))
	       qCritical() << "[Factory] Type factory" << namespaced << "already added !";
	  else
	       types.insert(namespaced, k.value());
	  
	  if (types.contains(k.key()))
	      qCritical() << "[Factory] Type factory" << namespaced << "already added !";
	  else
	       types.insert(k.key(), k.value());

	  QVariant var = k.value()->parseString(QString::null);
	  QString varName = var.typeName();
	  QString namespacedVarName = path + ":" + k.key();
	  if ( !variantTypeMap.contains(varName))
	  {
	       variantTypeMap.insertMulti(varName, k.key());
	       variantTypeMap.insertMulti(varName, namespacedVarName);
	  }
     }
     


     buildComponentLists();
     libraries.insert(path,lib);
     return true;
}




void ARCSFactory::unLoadLibrary(QString path)
{
     if (!libraries.contains(path))
     {
       qCritical() << "[Factory] Attempted to unload an unknown library" << path ;
	  return;
     }

     ARCSLibManager* lib = libraries[path];
     
     ARCSComponentMap acm = lib->getComponentMap();
     ARCSFamilyMap afm = lib->getFamilyMap();
     ARCSTypeMap atm = lib->getTypeMap();
     
     
     // parcourir la liste des composants : maj de native
     QHashIterator<QString, ARCSNativeComponent*> i(acm);
     
     while(i.hasNext())
     {
	  i.next();
	  QString namespaced = path + ":" + i.key();
	  
	  nativeFamily->removeFactory(namespaced);
	  nativeFamily->removeFactory(i.key());

	  nativeComponentLibraryMap.remove(namespaced);
	  nativeComponentLibraryMap.remove(i.key());
     }

     // parcours des familles
     QHashIterator<QString, ARCSAbstractFamily*> j(afm);

     while(j.hasNext())
     {
	  j.next();
	  QString namespaced = path + ":" + j.key();
	  families.remove(namespaced);
	  families.remove(j.key());
	  familyLibraryMap.remove(namespaced);
	  familyLibraryMap.remove(j.key());
     }

     // parcours des types
     QHashIterator<QString, ARCSTypeFactory*> k(atm);
     while (k.hasNext())
     {
	  k.next();
	  QString namespaced = path + ":" + k.key();
	  types.remove(namespaced);
	  types.remove(k.key());
	  typeLibraryMap.remove(namespaced);
	  typeLibraryMap.remove(k.key());

	  QVariant var = k.value()->parseString(QString::null);
	  variantTypeMap.remove(var.typeName());
     }

     buildComponentLists();
}




ARCSAbstractComponent* ARCSFactory::createComponent(QString type)
{
     if (! componentFamilyMap.contains(type))
     {
         ARCSLog::logError(ARCS_SOURCE,"component type " + type + " is unknown.");
          return 0;
     }

     QString familyName = componentFamilyMap[type];
     if (!families.contains(familyName))
     {
         ARCSLog::logError(ARCS_SOURCE,"family "+ familyName + " is unkown.");
          return 0;
     }

     ARCSAbstractFamily* aaf = families[familyName];
     

     if ( ! aaf->isInternal() ) 
     {
	  if (! familyLibraryMap.contains(familyName))
	  {
	       qCritical() << "[Factory] family:"  << familyName << "is unkown from family-library map." ; 
	       return 0;
	  }

	  QString libraryName = familyLibraryMap[familyName];
	  if (! libraries.contains(libraryName) )
	  {
	       qCritical() << "[Factory] library:" << libraryName << "is not loaded properly." ;
	       return 0;
	  }
	  
	  libraries[libraryName]->ref();
     }

     ARCSAbstractComponent* aac = aaf->instanciate(type);
     if (aac ==0)
	  return 0;
     
     aac->setFamily(familyName);
     aac->setType(type);
    
     // un rcent pb ici : ce qui tait retourn : aaf->instanciate
     // et non pas aac ?!? 
     return aac;
}


void ARCSFactory::destroyComponent(ARCSAbstractComponent* aac)
{
     QString familyName = aac->getFamily(); 

     if (familyName.isEmpty())
	  return ;

     if (!families.contains(familyName))
	  return ;

     ARCSAbstractFamily* aaf = families[familyName];

     if (aaf != 0)
	  aaf->destroy(aac);

     if (aaf->isInternal())
	  return ;
     
     if (! familyLibraryMap.contains(familyName))
	  return ;

     QString libraryName = familyLibraryMap[familyName];
     if (! libraries.contains(libraryName) )
	  return ;
     
     libraries[libraryName]->unref();
}



QString ARCSFactory::getVariantType(QVariant var)
{
     QStringList typesList = variantTypeMap.values(var.typeName());
     QString first,last;
     
     if (typesList.count() <=0 )
	  return QString::null;

     for (int i=0; i < typesList.count() ; i++)
     {
	  if (typesList[i].contains(":"))
	       last = typesList[i];
	  else
	       first = typesList[i];
     }

     if (!first.isEmpty())
	  if (types.count(first) <= 1)
	       return first;

     if (!last.isEmpty())
	  return last;
     
     return QString::null;
}


//! \todo factorize the function below with previous one
QString ARCSFactory::getInternalType(QString s)
{
    QStringList typesList = variantTypeMap.values(s);
    QString first,last;

    if (typesList.count() <= 0)
        return QString::null;

    for (int i=0; i < typesList.count() ; i++)
    {
     if (typesList[i].contains(":"))
          last = typesList[i];
     else
          first = typesList[i];
    }

    if (!first.isEmpty())
     if (types.count(first) <= 1)
          return first;

    if (!last.isEmpty())
     return last;

    return QString::null;
}




QVariant ARCSFactory::dataDeserialize(QString type, QString representation)
{
     if (!types.contains(type))
     {
	  qCritical() << "[Factory] Type" << type << "is unkown.";
	  return QVariant();
     }
     
     QVariant var = qVariantFromValue(types[type]->parseString(representation));

     if ( types[type]->isInternal())
	  return var;

     if (!typeLibraryMap.contains(type))
     {
	  qCritical() << "[Factory] Type" << type << "is unknown from type-library map.";
	  return var;
     }

     QString libraryName = typeLibraryMap[type];
     if (! libraries.contains(libraryName))
	  return var;

     libraries[libraryName]->ref();
     return var;
}


QString ARCSFactory::dataSerialize(QVariant var)
{     
     QString type = getVariantType(var);

     if (!types.contains(type))
     {
	  qCritical() << "[Factory] Type" << type << "is unkown.";
	  return QString::null;
     }

     return types[type]->toString(var);
}


QString ARCSFactory::getComponentLibrary(QString s)
{
    if (nativeComponentLibraryMap.contains(s))
        return nativeComponentLibraryMap[s];

    if (componentFamilyMap.contains(s))
    {
        QString family = componentFamilyMap[s];
        if (familyLibraryMap.contains(s))
            return familyLibraryMap[family];
    }

    return QString::null;
}

QStringList ARCSFactory::getLibraryComponents(QString s)
{
     if (! libraries.contains(s))
          return QStringList();

     QStringList res;
     res += libraries[s]->getComponentMap().keys();

     ARCSFamilyMap map = libraries[s]->getFamilyMap();

     QHashIterator<QString,ARCSAbstractFamily*> it(map);

     while (it.hasNext())
     {
          it.next();
          res += it.value()->factoryList();
     };

     return res;
}



QString ARCSFactory::getTypeLibrary(QString s)
{
     if (typeLibraryMap.contains(s))
	  return typeLibraryMap[s];

     return QString::null;
}


QString ARCSFactory::getFamilyLibrary(QString s)
{
     if (familyLibraryMap.contains(s))
	  return familyLibraryMap[s];

     return QString::null;
}
