/*
  name: lib/arcssheet.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcssheet.h>
#include <arcs/arcsinternaltypes.h>

#include <QMutableListIterator>
#include <QListIterator>
#include <iostream>
#include <QSet>

ARCSSheet ARCSSheet::null;


ARCSSheet::ARCSSheet(ARCSContext* ctx, QString n)
{
     name = n;
     activated = false;
     mutex = new QMutex();
     context = ctx;

}


ARCSSheet::ARCSSheet(const ARCSSheet& sheet)
{
     context = sheet.context;
     name = sheet.name;
     connections = sheet.connections;
     preconnects = sheet.preconnects;
     postconnects = sheet.postconnects;
     cleanups = sheet.cleanups;
     properties = sheet.properties;
     activated = false;
     mutex = new QMutex();
}


void ARCSSheet::reset()
{
     preconnects.clear();
     connections.clear();
     postconnects.clear();
     cleanups.clear();
     properties.clear();
}


QStringList ARCSSheet::getComponentList()
{
    QSet<QString> componentSet ;

    QListIterator<ARCSConnection> i(connections);
    while(i.hasNext())
    {
         ARCSConnection c = i.next();
         componentSet.insert(c.getSourceName());
         componentSet.insert(c.getDestinationName());
    }

    listComponentInit(preconnects,componentSet);
    listComponentInit(postconnects,componentSet);
    listComponentInit(cleanups,componentSet);
    return componentSet.toList();
}

void ARCSSheet::removeInitWithComponent(QList<ARCSInit>& lst,QString name)
{
    QMutableListIterator<ARCSInit> i(lst);
    while(i.hasNext())
    {
        i.next();
        if (i.value().getDestinationName() == name)
            i.remove();
    }

}


void ARCSSheet::removeComponent(QString name)
{
    QMutableListIterator<ARCSConnection> i(connections);

    while(i.hasNext())
    {
        ARCSConnection& connect = i.next();
        if (connect.getSourceName() == name)
        {
            i.remove();
        }
        else
        {
            if (connect.getDestinationName() == name)
            i.remove();
        }
    }
    removeInitWithComponent(preconnects,name);
    removeInitWithComponent(postconnects,name);
    removeInitWithComponent(cleanups,name);
}

void ARCSSheet::listComponentInit(QList<ARCSInit> &lst, QSet<QString> &set)
{
    QListIterator<ARCSInit> i(lst);
    while(i.hasNext())
    {
        ARCSInit init = i.next();
        set.insert(init.getDestinationName());
    }
}


void ARCSSheet::batchInit(QList<ARCSInit> & lst)
{
     QListIterator<ARCSInit> i(lst);
     while(i.hasNext())
     {
          ARCSInit init = i.next();
          if (init.isConstant())
               init.init(context->getConstant(init.getValue().value<ARCSConstant>()));
          else
          {
               if (init.isComponent())
               {
                    ARCSAbstractComponent* cmp = context->getComponent(init.getValue().value<ARCSComponent>());
                    if (cmp != 0)
                         init.init(cmp);
               }
               else
                    init.init();
          }
     }
}


void ARCSSheet::connect()
{
     QListIterator<ARCSConnection> i(connections);
     while(i.hasNext())
     {
          ARCSConnection c = i.next();
          c.connect();
     }
}


void ARCSSheet::disconnect()
{
     QListIterator<ARCSConnection> i(connections);
     while(i.hasNext())
     {
          ARCSConnection c = i.next();
          c.disconnect();
     }
}


ARCSConnection& ARCSSheet::addConnection(QString src, QString sgn, QString dst, QString slt, bool q,bool head)
{
     ARCSAbstractComponent* source = context->getComponent(src);
     ARCSAbstractComponent* destination = context->getComponent(dst);


     if (!source)
     {
          std::cerr << "Cannot find component named" << qPrintable(src) << std::endl;
     }

     
     ARCSConnection c(source, sgn, destination, slt, q);
     if (head)
     {
          connections.prepend(c);
          return connections.first();
     }

     connections.append(c);
     return connections.last();
}

ARCSConnection& ARCSSheet::getConnection(QString src, QString sgn, QString dst, QString slt)
{
     ARCSAbstractComponent* source = context->getComponent(src);
     ARCSAbstractComponent* destination = context->getComponent(dst);

     ARCSConnection c(source, sgn, destination, slt);

     int idx= connections.indexOf(c);
     if (idx != -1)
     {
          return connections[idx];
     }
     else
         return const_cast<ARCSConnection&>(ARCSConnection::null);
}


int ARCSSheet::getConnectionIndex(QString src, QString sgn, QString dst, QString slt)
{
    ARCSAbstractComponent* source = context->getComponent(src);
    ARCSAbstractComponent* destination = context->getComponent(dst);
    ARCSConnection c(source, sgn, destination, slt);
    return  connections.indexOf(c);
}

void ARCSSheet::swapConnections(int x, int y)
{
    if (x < connections.count()  && y < connections.count() && x >= 0 && y >=0  )
        connections.swap(x,y);
}


ARCSInit&  ARCSSheet::createInit(QList<ARCSInit> & lst, QString dst, QString slt, QString t, QString val)
{
     ARCSAbstractComponent* destination = context->getComponent(dst);
     QVariant value = context->createValue(t,val);
     ARCSInit i(destination, slt, value);
     lst.append(i);
     return lst.last();
}


void ARCSSheet::getConnections(QStringList& sources, QStringList& signls, 
                               QStringList& destinations, QStringList& slts)
{
     sources.clear();
     signls.clear();
     destinations.clear();
     slts.clear();

     QListIterator<ARCSConnection> i(connections);
     while(i.hasNext())
     {
          ARCSConnection c = i.next();

          sources << c.getSourceName();
          signls << c.getSignalName() ;
          destinations << c.getDestinationName();
          slts << c.getSlotName();
     }
}


void ARCSSheet::getInits(QList<ARCSInit> &lst, QStringList & destinations, 
                         QStringList & slts, QStringList & types, QStringList& values)
{
     destinations.clear();
     slts.clear();
     types.clear();
     values.clear();

     QListIterator<ARCSInit> i(lst);
     while(i.hasNext())
     {
          ARCSInit init = i.next();

          destinations << init.getDestinationName();
          slts << init.getSlotName();
          types << init.getValueType();
          values << init.getValueString();
     }
}


ARCSInit& ARCSSheet::getInit(QList<ARCSInit> &lst, QString dst, QString slt, QString t, QString val)
{
    ARCSAbstractComponent* destination = context->getComponent(dst);

    QVariant value = context->createValue(t,val);
    ARCSInit i(destination, slt, value);

    int idx= lst.indexOf(i);
    if (idx != -1)
         return lst[idx];
    else
    {
         return ARCSInit::null;
    }
}


void ARCSSheet::swapInits(QList<ARCSInit> &lst,int x, int y)
{
    if (x < lst.count() && y < lst.count() && x >=0 && y >=0)
        lst.swap(x,y);
}
