/*
  name: tools/builder/arcsbuilder.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http: *www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http: *www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/



/*! \page arcsbuilder arcsbuilder manpage
 *<H2>NAME</H2>
 *<B>arcsbuilder</B> helper tool to build component libraries for ARCS.
 *<H2>SYNOPSIS</H2>
 *<B>arcsbuilder </B> XML_FILE [COMPONENTS_PATH]
 *<H2>DESCRIPTION</H2>
 *<B>arcsbuilder</B> builds component libraries required by an XML description of an application
 *(<I>XML_FILE</I>).
 *A second parameter (<I>COMPONENTS_PATH</I>) may be needed in order to set the path where component libraries can be
 *found. If not given, then the program will fallback to the
 *<I>ARCSBUILDPATH</I>
 *environment variable (if available) that have the same use.
 *<H2>AUTHOR</H2>
 *Jean-Yves Didier
 */


#include <arcs/arcsxmlhandler.h>
#include <arcs/arcsapplicationcomponent.h>
#include <arcs/arcsfactory.h>
#include <QCoreApplication>
#include <QString>
#include <iostream>
#include <iomanip>
#include <cstdlib>

#include <QMap>
#include <QStringList>
#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QRegExp>
#include <QTextStream>
#include <QProcess>
#include <QXmlQuery>

#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif



typedef struct {
     QString path;
     bool libmaker;
} LibraryPath ;

LibraryPath libraryPath(QString p, bool b)
{
     LibraryPath res ;
     res.path = p ;
     res.libmaker = b;
     return res;
}


QMap<QString,LibraryPath> libraryDirBounds;

/*! \brief Strips the library name from a given path
 *
 */
QString getLibraryName(QString libpath)
{
     QString lname = libpath.right(libpath.length() - libpath.lastIndexOf(QDir::separator())-1 );
     lname = lname.left(lname.lastIndexOf('.'));
     if (lname.startsWith("lib"))
     {
       lname = lname.right(lname.length()-3);
     }

     return lname;
}


void exploreProjectFile(QString profile)
{
      /* First : strip the name of the pro file*/
     QFileInfo fi(profile) ;
     QString name = fi.baseName();
     QString path = fi.absolutePath();

      /* Second open the file and detect lines
      * TARGET = something */
     QString target;

     QFile filepro(profile);
     filepro.open(QIODevice::ReadOnly);
     QTextStream ts(&filepro);
     QRegExp linematcher("^\\s*TARGET\\s*=\\s*(\\w+)\\W*$");


     QString line;
     do
     {
       line = ts.readLine();
       if (linematcher.exactMatch(line))
       {
            target = linematcher.cap(1);
       }
     }
     while ( !line.isNull());

      /* select what to put inside mapper */
     if (target.isEmpty())
     {
       std::cout << std::setw(15) << qPrintable(name) << " => " << qPrintable(path) << std::endl;
       libraryDirBounds[name] = libraryPath(path, false);
     }
     else
     {
       std::cout << std::setw(15) << qPrintable(target) << " => " << qPrintable(path) << std::endl;
       libraryDirBounds[target] = libraryPath(path,false);
     }
}


bool launchProcess(QString processName, QString path)
{
     QProcess pr;
     pr.setWorkingDirectory(path);
     std::cout << "     " << qPrintable(processName);
     pr.start(processName);
     while (pr.state() == QProcess::Running)
     {
          std::cout << "." ;
          std::cout.flush();

#ifdef WIN32
          Sleep(1000);
#else
          sleep(1);
#endif
     }
     std::cout << std::endl;
     pr.waitForFinished();
     return (pr.exitStatus() == QProcess::NormalExit && pr.exitCode() == 0);
}


void buildBounds(QString path)
{
     unsigned int i;
     QDir d(path);

      /* Make sure path is '/' terminated*/
     if (path.at(path.length()-1) != QDir::separator())
       path += QDir::separator();

      /* First pass : scan all .pro files*/
     QStringList profiles = d.entryList(QStringList("*.pro"));
     for (i=0; i < profiles.count() ; i++)
       exploreProjectFile(path + profiles[i]);

     if( profiles.count() == 0)
     {
          std::cout << "Storing alxfiles" << std::endl;
          QStringList alxfiles = d.entryList(QStringList("*.alx"));
          for (i=0; i < alxfiles.count() ; i++)
          {
               std::cout << qPrintable(d.dirName()) << std::endl;
              libraryDirBounds[d.dirName()] = libraryPath( path, true );
         }
     }

      /* Second pass : scan all directories */
     QStringList dirs = d.entryList(QDir::Dirs);
     for (i=0; i < dirs.count() ; i++)
     {
        /* filter out directories with leading '.'*/
       if (dirs[i].at(0) != '.' )
            buildBounds(path + dirs[i]);
     }
}


bool buildLibrary(QString path,bool libmaker=false)
{
     if (libmaker)
          if (!launchProcess("arcslibmaker",path))
               return false;
     if (!launchProcess("qmake",path))
          return false;
     if (!launchProcess("make",path))
          return false;
     return true;
}


int buildApplication(QStringList libs, QString path)
{
     int success = 0;

     std::cout << "Looking for bindings between libs and directories" << std::endl;
     buildBounds(path);

     for (int i=0; i < libs.count() ; i++)
     {
       QString libname = getLibraryName(libs[i]);
       std::cout << "Trying to build library " << qPrintable(libname) << std::endl;
       QString dir = libraryDirBounds[libname].path;
       if (!dir.isEmpty())
       {
            if (buildLibrary(dir,libraryDirBounds[libname].libmaker))
            {
              std::cout << std::setw(10) << " " << " Success !" << std::endl;
              success++;
            }
            else
              std::cout << std::setw(10) << " " <<  "Failed !" << std::endl;


       }
       else
            std::cout << std::setw(10) << " " << "Failed !"  << std::endl;
     }
     return success;
}




void usage(char* name)
{
     std::cerr << "Usage: " << name << " application_file.xml [component/sources/path/]" << std::endl;
     std::cerr << "Where: \"application_file.xml\" is an application file ;" << std::endl;
     std::cerr << "       \"component/sources/path/\" is an optional source path. " << std::endl;
}


int main(int argc,char* argv[])
{
     QString path;
     QCoreApplication app(argc,argv);
     if ((argc == 1) || (argc > 3))
     {
          usage(argv[0]);
          return 1;
     }

     char* buildpath = getenv("ARCSBUILDPATH");

     if (buildpath == NULL)
          std::cerr << "ARCSBUILDPATH environment variable not found." << std::endl;
     else
          path = QString(buildpath);

     if (buildpath == NULL && argc < 3)
     {
          usage(argv[0]);
          return 1;
     }

     if (argc == 3 )
          path = QString(argv[2]);


     QXmlQuery query;

     query.bindVariable("file",QVariant(QDir::currentPath() + QDir::separator() + QString(argv[1])));
     query.setQuery("doc($file)/application/context/libraries/library/string(data(@path))");
     QStringList libList;
     if (query.evaluateTo(&libList))
     {
          int res = buildApplication(libList,path);
          if (res != libList.count())
               std::cerr << "Failed to build some libraries required by your application." << std::endl;
          else
               std::cout << "All libraries successfully built." << std::endl;
     }
     else
     {
          std::cerr << "Failed to load file " << argv[1] << std::endl;
          return 1;
     }

     return 0;
}
