/*
  name: tools/wizard/arcstablauncher.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "arcstablauncher.h"
#include "ui_arcstablauncher.h"

#include <QFileDialog>
#include <QXmlQuery>
#include <QDomDocument>
#include <QFile>
#include <QTextStream>
#include <QProcess>
#include <QMessageBox>
#include <QFileSystemModel>
#if QT_VERSION < 0x050000
#include <QColorGroup>
#endif
#include <iostream>


ARCSTabLauncher::ARCSTabLauncher(QWidget *parent) :
    QTabWidget(parent),
    ui(new Ui::ARCSTabLauncher)
{
    ui->setupUi(this);
    process =0;
    actualMode=0;
    ui->inputProfileLineEdit->setFilter(QStringList("*.xml"));
    ui->applicationLineEdit->setFilter(QStringList("*.xml"));
    ui->outputProfileLineEdit->setFilter(QStringList("*.xml"));
    ui->outputProfileLineEdit->setFileExists(false);
}

ARCSTabLauncher::~ARCSTabLauncher()
{
    delete ui;
}

void ARCSTabLauncher::on_applicationBrowseButton_clicked()
{
     QString res = QFileDialog::getOpenFileName(this, "Select an application file",QString(),"*.xml");

     if (res.isEmpty())
          return;

     ui->applicationLineEdit->setText(res);
     updateApplication(res);
}


void ARCSTabLauncher::updateApplication(QString s)
{
     QXmlQuery query;
     query.bindVariable("file",QVariant(s));
     query.setQuery("doc($file)/application/data(@mode)");
     QString mode;
     if (query.evaluateTo(&mode))
     {
          mode = mode.toLower().trimmed();
          if (mode == "base")
              appMode=0;
          if (mode == "event")
              appMode=1;
          if (mode == "thread")
              appMode=2;
          if (mode == "threadevent")
              appMode=3;
          if (mode == "gui")
              appMode=4;
           actualMode = appMode;
           ui->applicationMode->setCurrentIndex(appMode);
      }

     appContext.clear();
     query.setQuery("doc($file)/application/context/constants/constant/string-join((data(@id),data(@type),data(node())),':')");
     QStringList answer;
     if (query.evaluateTo(&answer))
          for (int i=0; i < answer.count(); i++)
              appContext.createConstant(answer[i].section(':',0,0),answer[i].section(':',1,1),answer[i].section(':',2));
     actualContext = appContext;
     updateTable(appContext);
     ui->inputProfileLineEdit->setText(QDir::currentPath());

}

void ARCSTabLauncher::on_inputProfileBrowseButton_clicked()
{
     QString res = QFileDialog::getOpenFileName(this, "Select a profile",QString(),"*.xml");

     if (res.isEmpty())
          return;

     ui->inputProfileLineEdit->setText(res);
     updateInputProfile(res);

}


void ARCSTabLauncher::updateInputProfile(QString s)
{
     ARCSEngineContext proContext;
     QXmlQuery query;
     query.bindVariable("file",QVariant(s));
     query.setQuery("doc($file)/profile/constant/string-join((data(@id),data(@type),data(node())),':')");
     QStringList answer;
     if (query.evaluateTo(&answer))
         for (int i=0; i < answer.count(); i++)
             proContext.createConstant(answer[i].section(':',0,0),answer[i].section(':',1,1),answer[i].section(':',2));

     actualContext = appContext.merge(proContext);
     updateTable(actualContext);

}


void ARCSTabLauncher::on_outputProfileBrowseButton_clicked()
{
     QString res = QFileDialog::getSaveFileName(this, "Select a profile",QString(),"*.xml");

     if (res.isEmpty())
          return;

     ui->outputProfileLineEdit->setText(res);
}


void ARCSTabLauncher::updateTable(ARCSEngineContext aec)
{
    ui->constantsTable->clearContents();
    ui->constantsTable->setRowCount(aec.getConstantNumber());
    QStringList ids=aec.getConstants();

    for (int i=0; i < ids.count();i++ )
    {
        QTableWidgetItem* cName= new QTableWidgetItem(ids[i]);
        cName->setFlags(Qt::NoItemFlags);
        QTableWidgetItem* tName= new QTableWidgetItem(aec.getConstantType(ids[i]));
        tName->setFlags(Qt::NoItemFlags);

        ui->constantsTable->setItem(i,0,cName);
        ui->constantsTable->setItem(i,1,tName);
        ui->constantsTable->setItem(i,2,new QTableWidgetItem(aec.getConstantValue(ids[i])));
    }
}

ARCSEngineContext ARCSTabLauncher::constantsFromTable()
{
    ARCSEngineContext res;
    for (int i=0; i < ui->constantsTable->rowCount(); i++)
        res.createConstant(ui->constantsTable->item(i,0)->text(),
                           ui->constantsTable->item(i,1)->text(),
                           ui->constantsTable->item(i,2)->text());

    return res;
}


void ARCSTabLauncher::on_launchButton_clicked()
{
    const char* params[] = { "-b", "-e", "-t" , "-te", "-g" };

    ARCSEngineContext aec= actualContext.diff(constantsFromTable());

    ui->outputArea->appendHtml("<hr/>");
    ui->launchButton->setEnabled(false);
    this->setCurrentIndex(1);

    process=new QProcess();

    connect(process,SIGNAL(readyReadStandardError()),this,SLOT(readStdErr()));
    connect(process,SIGNAL(readyReadStandardOutput()),this,SLOT(readStdOut()));
    connect(process,SIGNAL(finished(int,QProcess::ExitStatus)),this,SLOT(processFinished(int,QProcess::ExitStatus)));

    QString parms;
    QStringList sl = aec.getConstants();

    for (int i=0; i < aec.getConstantNumber() ;i++)
    {
        parms += sl[i] + "=" + aec.getConstantValue(sl[i]);
        if (i != aec.getConstantNumber() -1)
            parms+= ":";
    }


    QStringList arguments;
    arguments << "-l" << "2" ;
    arguments << params[actualMode];
    if (!parms.isEmpty())
    {
        arguments << "-d" ;
        arguments << parms;
    }

    if (ui->inputProfileLineEdit->hasAcceptableInput())
         arguments << "-p" << ui->inputProfileLineEdit->text() ;
    if (ui->outputProfileLineEdit->hasAcceptableInput())
         arguments << "-o" << ui->outputProfileLineEdit->text() ;

    if (!ui->applicationLineEdit->hasAcceptableInput())
         return ;
    arguments << ui->applicationLineEdit->text();


    process->start("arcsengine",arguments);
    ui->killButton->setEnabled(true);
    ui->restartButton->setEnabled(false);
    ui->launchButton->setEnabled(false);
}



void ARCSTabLauncher::processFinished(int code,QProcess::ExitStatus)
{
    delete process;
    process = 0;
    ui->launchButton->setEnabled(true);
    ui->outputArea->appendHtml(QString("<br/><font color=\"navy\"><b>Application finished with code ")+ QString::number(code) +"</b></font>");
    ui->killButton->setEnabled(false);
    ui->launchButton->setEnabled(true);
    ui->restartButton->setEnabled(true);
}



void ARCSTabLauncher::readStdErr()
{
    QString err(process->readAllStandardError());
    //QRegExp regexp("\033(.*)m");
    err.replace("\033[32;1m","<font color=\"green\">");
    err.replace("\033[34;1m","<font color=\"blue\">");
    err.replace("\033[33;1m","<font color=\"yellow\">");
    err.replace("\033[31;1m","<font color=\"red\">");
    err.replace("\033[0m","</font>");
    err.replace("\n","<br/>");
    ui->outputArea->appendHtml(err);
}

void ARCSTabLauncher::readStdOut()
{
    QString out(process->readAllStandardOutput());
    out.replace("\033[32;1m","<font color=\"green\">");
    out.replace("\033[34;1m","<font color=\"blue\">");
    out.replace("\033[33;1m","<font color=\"yellow\">");
    out.replace("\033[31;1m","<font color=\"red\">");
    out.replace("\033[0m","</font>");
    out.replace("\n","<br/>");
    ui->outputArea->appendHtml(out);
}

void ARCSTabLauncher::on_restoreValuesButton_clicked()
{
    updateTable(actualContext);
}

void ARCSTabLauncher::on_saveProfileButton_clicked()
{
     QString res = QFileDialog::getSaveFileName(this, "Select a profile",QString(),"*.xml");

     if (res.isEmpty())
          return;

     ARCSEngineContext aec= actualContext.diff(constantsFromTable());

     QDomDocument doc;
     QDomElement profile = doc.createElement("profile");
     doc.appendChild(profile);

     QStringList sl = aec.getConstants();

     for (int i=0; i < aec.getConstantNumber() ;i++)
     {
          QDomElement constant = doc.createElement("constant");
          constant.setAttribute("id",sl[i]);
          constant.setAttribute("type",aec.getConstantType(sl[i]));
          constant.appendChild(doc.createTextNode(aec.getConstantValue(sl[i])));
          profile.appendChild(constant);
     }

     QFile file(res);
     if (!file.open(QIODevice::WriteOnly))
     {
          QMessageBox::critical(this,"Failed to save profile", "Could not save profile in file "+res );
          return ;
     }
     QTextStream ts(&file);
     doc.save(ts,4);
     file.close();
}

void ARCSTabLauncher::on_killButton_clicked()
{
    if (process != 0)
         process->kill();
}

void ARCSTabLauncher::on_restartButton_clicked()
{
    on_launchButton_clicked();
}


void ARCSTabLauncher::on_applicationMode_currentIndexChanged(int index)
{
    actualMode = index;
}

void ARCSTabLauncher::on_applicationLineEdit_editingFinished()
{
    updateApplication(ui->applicationLineEdit->text());
}

void ARCSTabLauncher::on_inputProfileLineEdit_editingFinished()
{
     updateInputProfile(ui->inputProfileLineEdit->text());
}
