// ARCS.js : 27/6/2014 16:00
// handling dependencies


//"use strict";


/** 
 * Main source: describes all the methods needed by the ARCS engine
 * @file
 */


/**
 * Defines all elements needed for Augmented Reality Component System
 * @namespace
 */
var ARCS =  ARCS || {};


/******************************************************************************
 * Helper functions to determine environment
 * ***************************************************************************/


/**
 * @return {boolean} true if ARCS is run in a node.js environment
 */
ARCS.isInNode = function () {
    return (typeof require === 'function' && require.resolve);
};

/**
 * @return {boolean} true if ARCS is run with require.js
 */
ARCS.isInRequire = function () {
    return (typeof define === 'function' && define.amd);
};


/******************************************************************************
 * Component implementation
 * ***************************************************************************/


/** 
 * Defines main traits of components in a namespace regrouping important methods
 * 
 * @namespace 
 */
ARCS.Component = {
    /** Error message */
    SourceIsNotComponent : {message : "Source is not a component"},
    /** Error message */
    UndefinedSignal : {message : "Signal is not defined"},
    /** Error message */
    UndefinedSlot : {message : "Slot is not defined"},
    /**
     * External constructor: give component traits to any constructor.
     * 
     * Component traits are the following: 
     * <ul>
     * <li>Slot functions listed in an array;</li>
     * <li>A signal list described in an array;</li>
     * <li>A method returning the slot list;</li>
     * <li>A method returnung the signal list;</li>
     * <li>An emit method, to trigger signals by their names;</li>
     * <li>A slot method to cast an internal method to a slot;</li>
     * <li>A signal mehtod to register a possible signal.</li>
     * </ul>
     * @param name {string} Class name to transform to a component
     * @param sltList {string[]} names of functions designated as slots, may be empty.
     * @param sgnList {string[]} names of functions designated as signals, may be empty.
     */
    create : function (name, sltList, sgnList) {
        if (name.prototype === undefined) {
            console.error("Cannot create such a component");
            return 0;
        }

        name.prototype.slots = [];
        name.prototype.signals = {};
        name.slotList = function () {
            return name.prototype.slots;
        };
        name.prototype.slotList = function () {
            return name.prototype.slots;
        };
        name.prototype.signalList = function () {
            var res = [], i;
            for (i in name.prototype.signals) {
                res.push(i);
            }
            return res;
        };
        name.signalList = function () {
            return name.prototype.signalList();
        };
        name.prototype.emit = function (signal) {
            var slt, func, obj;
            var args = Array.prototype.slice.call(arguments,1);
            for (slt in this.signals[signal]) {
                func = this.signals[signal][slt].func;
                obj = this.signals[signal][slt].obj;
                func.apply(obj, args);
            }
        };
        name.slot = function (slot, func) {
            var i;
            if (slot instanceof Array) {
                for (i = 0; i < slot.length; i++) {
                    name.prototype.slots.push(slot[i]);
                }
            } else {
                name.prototype.slots.push(slot);
                if (func !== undefined) {
                    name.prototype[slot] = func;
                }
            }
        };
        name.signal = function (signal) {
            var i;
            if (signal instanceof Array) {
                for (i = 0; i < signal.length; i++) {
                    name.prototype.signals[signal[i]] = 1;
                }
            } else {
                name.prototype.signals[signal] = 1;
            }
        };

        // code for returning component, and or completing its definition
        if (sltList !== undefined) {
            name.slot(sltList);
        }

        if (sgnList !== undefined) {
            name.signal(sgnList);
        }
        return name;
    },
    /** 
     * Checks if the given prototype has traits of a component
     * @param name {string} name of the prototype
     */
    check : function (name) {
        if (name.prototype === undefined) {
            return false;
        }
        if (name.prototype.signals === undefined ||
                name.prototype.slots === undefined) {
            return false;
        }
        return true;
    },
    /**
     * Connects two different components by using their signal and slots
     * @param source {object} component sending data
     * @param signal {string} name of the signal to connect
     * @param destination {object} component receiving data
     * @param slt {string} name of the slot to connect
     */
    connect : function (source, signal, destination, slt) {
        var orig, p;
        // here we can perform various checks.
        if (source.signals === undefined) {
            throw ARCS.Component.SourceIsNotComponent;
        }
        if (source.signals[signal] === undefined) {
            throw ARCS.Component.UndefinedSignal;
        }
        if (destination[slt] === undefined) {
            throw ARCS.Component.UndefinedSlot;
        }
        // we must also check if the signals dispose of their own implementation
        if (!source.hasOwnProperty('signals')) {
            // otherwise, we should clone it so that each component dispose of its 
            // own signal copy.
            orig = source.signals;
            source.signals = {};
            for (p in orig) {
                source.signals[p] = [];
            }
        }
        source.signals[signal].push({obj: destination, func: destination[slt]});
    },
    /**
     * Diconnects a signal/slot connection between two components
     * @param source {object} component sending data
     * @param signal {string} name of the signal to connect
     * @param destination {object} component receiving data
     * @param slt {string} name of the slot to connect
     */
    disconnect : function (source, signal, destination, slt) {
        var i;
        for (i = 0; i < source.signals[signal].length; i++) {
            if (source.signals[signal][i].obj === destination) {
                if (source.signals[signal][i].func === destination[slt]) {
                    source.signals[signal].splice(i, 1);
                    i--;
                }
            }
        }
    },
    /**
     * Invokes a specific slot of a given component
     * @param destination {object} component upon which invocation is performed
     * @param slt {string} name of the slot to invoke
     * @param value {mixed} value to input
     */
    invoke : function (destination, slt, value) {
        if (destination[slt] === undefined) {
            throw ARCS.Component.UndefinedSlot;            
        }
        
        
        var func = destination[slt];
        func.apply(destination, value);
    },
    /** 
     * Specific hook that can be called when initializing a component
     * @param component {object} prototype of the component
     * @param obj {object} the actual object
     */
    config : function (component, obj) {
        if (typeof component.config === 'function') {
            component.config(obj);
        }
    }
};

ARCS.Context = function( props ) { 
    for (p in props) {
        if (props.hasOwnProperty(p)) {
            this[p] = props[p];
        }
    }
};

ARCS.Context.prototype.getComponentList = function () {
    var list = Object.keys(this);
    var i;
        
    for (i = 0; i < list.length; i++) {
        if ( ! this.hasOwnProperty(list[i])) {
            list.splice(i--,1);
        }
    }
    return list;
};

ARCS.Context.prototype.getComponent = function (cName) {
    if (!this.hasOwnProperty(cName))
            return undefined;
    return this[cName].instance;
};

ARCS.Context.prototype.getComponentName = function (cmp) {
    var i, keys;
    keys = this.getComponentList();
           
    for(i = 0; i < keys.length; i++) {
        if (this[keys[i]].instance === cmp) {
                return keys[i];
        }
    }
        
    return undefined;
};

ARCS.Context.prototype.toJSON = function () {
    var res = {}, p;
    
    for (p in this) {
        if (this.hasOwnProperty(p)) {
            res[p] = { type: this[p].type, value: this[p].value };
        }
    }
    return res;
};


/******************************************************************************
 * Invocation implementation
 * ***************************************************************************/
/**
 * Defines an invocation
 * @param destination {object} component on which to perform invocation
 * @param slot {string} name of the slot 
 * @param value {mixed} value passed to the invoked slot
 * @constructor
 */
ARCS.Invocation = function (destination, slot, value) {
    this.getDestination = function () {
        return destination;
    };
    
    this.getSlot = function () {
        return slot;
    };
    
    this.getValue = function () {
        return value;
    };
    
    this.invoke = function () {
        var func = destination[slot];
        if (func === undefined) {
                console.error("Undefined slot %s of component %s", slot, destination);
                return;
        } 
        func.apply(destination, value);
    };
};
/**
 * Helper function that casts an invocation from a description 
 * @param obj {object} a raw description of the invocation
 * @param context {object} the context in which this invocation takes place.
 * @return an invocation
 */
ARCS.Invocation.cast = function (obj, context) {
    return new ARCS.Invocation(context[obj.destination].instance, obj.slot, obj.value);
};

/*ARCS.Invocation.revert = function(obj, context) {
    return {
        destination: context
        
    };
    
};*/


ARCS.Invocation.PRE_CONNECTION = 0;
ARCS.Invocation.POST_CONNECTION = 1;
ARCS.Invocation.CLEAN_UP = 2;
/******************************************************************************
 * Connection implementation
 * ***************************************************************************/
/** 
 * Defines a connection between two components
 * @param source {object} component at the source
 * @param signal {string} name of the signal emitting data
 * @param destination {object} component at the destination 
 * @param slot {string} name of the signal receiving data
 * @class
 */
ARCS.Connection = function (source, signal, destination, slot) {
    /**
     * Connects two components as described in this object
     */
    this.connect = function () {
        ARCS.Component.connect(source, signal, destination, slot);
    };
    /**
     * Disconnects a signal/slot connection between the two components
     * described in this object.
     */
    this.disconnect = function () {
        ARCS.Component.disconnect(source, signal, destination, slot);
    };
    
    this.getSource = function() {
        return source; 
    };
    
    this.getDestination = function () {
        return destination;
    };
    
    this.getSlot = function ()  {
        return slot;
    };
    
    this.getSignal = function () {
        return signal;
    };   
};
/**
 * Helper function that casts a connection from a description 
 * @param obj {object} a raw description of the connection
 * @param context {object} the context in which this connection takes place.
 * @return a connection
 */
ARCS.Connection.cast = function (obj, context) {
    return new ARCS.Connection(context[obj.source].instance, obj.signal,
                                context[obj.destination].instance, obj.slot);
};

/******************************************************************************
 * Sheet implementation
 * ***************************************************************************/
/**
 * Constructs a sheet
 * @param context {object} a context object
 * @class
 * @classdesc A Sheet is an operationnal configuration in an application. It 
 * contains many things: multiple sets of {@link ARCS.Invocation} 
 * performed at different times
 * and a set of {@link ARCS.Connection}. Sheets have two high level operations:
 * activation and deactivation. 
 */
ARCS.Sheet = function (context) {
    //var context = ctx;
    var preconnections = [], postconnections = [], cleanups = [], connections = [],
        invokePreconnections, invokePostconnections, invokeCleanups, 
        connect, disconnect, getComponentName,
        preCount = 0, postCount = 0, cleanCount = 0, connCount = 0;
        
    invokePreconnections = function () {
        var i;
        for (i = 0; i < preconnections.length; i++) {
            preconnections[i].invoke();
        }
    };
    invokePostconnections = function () {
        var i;
        for (i = 0; i < postconnections.length; i++) {
            postconnections[i].invoke();
        }
    };
    invokeCleanups = function () {
        var i;
        for (i = 0; i < cleanups.length; i++) {
            cleanups[i].invoke();
        }
    };
    connect = function () {
        var i;
        for (i = 0; i < connections.length; i++) {
            connections[i].connect();
        }
    };
    disconnect = function () {
        var i;
        for (i = 0; i < connections.length; i++) {
            connections[i].disconnect();
        }
    };
    
    this.setContext = function (ctx) {
        context = ctx;
    };
    /**
     * Activates this sheet. Pre-connection invocations are peformed, then 
     * connections are established and post-connection invocations are finally 
     * performed.
     */
    this.activate = function () {
        invokePreconnections();
        connect();
        invokePostconnections();
    };
    /**
     * Deactivates this sheet. Connections are removed and then cleanup invocations
     * are performed.
     */
    this.deactivate = function () {
        disconnect();
        invokeCleanups();
    };
    
    this.addPreConnection = function (obj) {
        var pre = ARCS.Invocation.cast(obj, context);
        pre.id = preCount++;
        preconnections.push(pre);
        return pre.id;
    };
    
    this.addPostConnection = function (obj) {
        var post = ARCS.Invocation.cast(obj, context);
        post.id = postCount++;
        postconnections.push(post);
        return post.id;
    };


    this.addCleanup = function (obj) {
        var cleanup = ARCS.Invocation.cast(obj, context);
        cleanup.id = cleanCount++;
        cleanups.push(cleanup);
        return cleanup.id;
    };
    
    this.addConnection = function (obj) {
        var connection = ARCS.Connection.cast(obj, context);
        connection.id = connCount++;
        connections.push(connection);
        return connection.id;
    };

    
    var removeItem = function(id, tab) {
        var i = tab.length;
        
        while ( i-- && tab[i].id !== id );
        
        if (i >= 0) {
            tab.splice(i,1);
        } else {
            console.warn("Could not remove data with id", id);
        }
    };
    
    this.removePreConnection = function (id) {
        removeItem(id, preconnections);
    };
    
    this.removePostConnection = function (id) {
        removeItem(id, postconnections);
    };
    
    this.removeCleanup = function (id) {
        removeItem(id, cleanups);
    };
    
    var changeItem = function(id, value, tab) {
        var i = tab.length;        
        while ( i-- && tab[i].id !== id );
        if (i >= 0) {
            tab[i].value = value;
        }
    };
    
    this.changePreConnection = function (id, value) {
        changeItem(id, value, preconnections);
    };
    
    this.changePostConnection = function (id, value) {
        changeItem(id, value, postconnections);
    };
    
    this.changeCleanup = function (id, value) {
        changeItem(id, value, cleanups);
    };
    
    this.removeConnection = function (id) {
        removeItem(id, connections);
    };
    
    
    var swapItems = function (id1, id2, tab) {
        var item;
        
        var i = tab.length, j = tab.length;
        
        while( i-- && tab[i].id !== id1 ) ;
        while( j-- && tab[j].id !== id2 ) ;

        if (i >= 0 && j >= 0) {
            item = tab[i];
            tab[i] = tab[j];
            tab[j] = item;
            tab[i].id = id1;
            tab[j].id = id2;
        }
    };
    
    this.swapConnections = function (id1, id2) {
        swapItems(id1, id2, connections);
    };
    
    this.swapCleanups = function (id1, id2) {
        swapItems(id1, id2, cleanups);
    };
    
    this.swapPreConnections = function (id1, id2) {
        swapItems(id1, id2, preconnections);
    };
    
    this.swapPostConnections = function (id1, id2) {
        swapItems(id1, id2, postconnections);
    };

    /**
     * Imports a structure object describing the content of a sheet.
     * @param object {object} structured object describing sheet's content.
     */
    this.import = function (object) {
        var i = 0, castInvocation = ARCS.Invocation.cast, castConnection = ARCS.Connection.cast;
        for (i = 0; i < object.preconnections.length; i++) {
            preconnections.push(castInvocation(object.preconnections[i], context));            
        }
        for (i = 0; i < object.postconnections.length; i++) {
            postconnections.push(castInvocation(object.postconnections[i], context));
        }
        for (i = 0; i < object.cleanups.length; i++) {
            cleanups.push(castInvocation(object.cleanups[i], context));
        }
        for (i = 0; i < object.connections.length; i++) {
            connections.push(castConnection(object.connections[i], context));
        }
    };
    
    var revertInvocation = function (obj) { 
        return {
            destination: context.getComponentName(obj.getDestination()),
            slot: obj.getSlot(),
            value: obj.getValue()
         };
    };
    
    var revertConnection = function (obj) {
        return {
            source: context.getComponentName(obj.getSource()),
            signal: obj.getSignal(),
            destination: context.getComponentName(obj.getDestination()),
            slot: obj.getSlot()
        }; 
    };
    
    this.toJSON = function () {
        var preconns = [];
        var postconns = [];
        var conns = [];
        var cleans = [];
        
        var i;        
        for (i = 0; i < connections.length; i++) {
            conns.push(revertConnection(connections[i]))
        }
        for (i = 0; i < preconnections.length; i++) {
            preconns.push(revertInvocation(preconnections[i]))
        }
        for (i = 0; i < postconnections.length; i++) {
            postconns.push(revertInvocation(postconnections[i]))
        }
        for (i = 0; i < cleanups.length; i++) {
            cleans.push(revertInvocation(cleanups[i]))
        }
                
        return  {
            preconnections : preconns,
            postconnections : postconns,
            connections: conns,
            cleanups: cleans           
        };
    };
};

/******************************************************************************
 * Statemachine implementation
 * ***************************************************************************/
/**
 * Describes a statemachine
 * @param obj {object} an object describing a state machine. If obj is empty then the statemachine is empty
 * @class
 */
ARCS.Statemachine = function (obj) {
    // dynamic construction: properties are initial state that have properties 
    // that are tokens and value that are the final state
    var initial = "", final = "", transitions = {}, currentState = "";

    if (obj !== undefined) {
        initial = obj.initial;
        final = obj.final;
        transitions = obj.transitions;
        currentState = "";
    }
    /**
     * Sets the initial state of the statemachine
     * @param string {string} name of the initial state
     */
    this.setInitialState = function (string) {
        initial = string;
        currentState = initial;
    };
    /**
     * Sets the final state of the statemachine
     * @param string {string} name of the final state
     */
    this.setFinalState = function (string) { final = string; };
    /**
     * Adds a transition to the state machine
     * @param start {string} name of the state at the beginning of the transition
     * @param token {string} name of the token triggering the transition
     * @param end {string} name of the state reached at the end of the transition
     */
    this.addTransition = function (start, token, end) {
        transitions[start][token] = end;
    };
    /**
     * Gives a token to the statemachine. According to its list of transitions
     * and the current state, it may trigger a transition
     * @param token {string} name of the token
     */
    this.setToken = function (token) {
        if (transitions[currentState] !== undefined) {
            if (transitions[currentState][token] !== undefined) {
                currentState = transitions[currentState][token];
                this.emit('requestSheet', currentState);
                if (currentState === final) {
                    this.emit('requestTermination');
                }
            }
        }
    };
    /**
     * Sets transitions from a list of transitions
     * @param obj {object[]} list of transitions
     */
    this.setTransitions = function (obj) { transitions = obj; };
    /**
     * Initialize and starts the statemachine, setting its current state to 
     * the initial state (by default, it is the departure of the first transition
     */
    this.start = function () {
        currentState = initial;
        this.emit('requestSheet', currentState);
    };
};


ARCS.Component.create(ARCS.Statemachine);
ARCS.Statemachine.slot("setToken");
ARCS.Statemachine.signal("requestSheet");
ARCS.Statemachine.signal("requestTermination");

/******************************************************************************
 * Application implementation
 * ***************************************************************************/
/**
 * Creates an application runnable by the ARCS engine.
 * @class
 * @classdesc The application is the master class of the ARCS engine. 
 * It is initialized using a structured object (possibly described in JSON, 
 * see {@link ARCS.Application#import}) 
 * to load all external scripts describing components, instanciate
 * all components and then start the application
 */
ARCS.Application = function () {
    var context = new ARCS.Context(),
        sheets = {},
        controller = {},
        libraries = [],
        dependencies = [],
        factories = {},
        self = this,
        autoStart = false,
        currentSheet = "",
        loadLibraries,
        preProcess,
        checkFactories;

    factories.StateMachine = ARCS.Statemachine;
    

    loadLibraries = function () {
        var i;
        // we will use different instances of require either the one of node 
        // or the one from require.js
        ARCS.Application.currentApplication = self;

        // in node.js, we have a synchronous load
        if (ARCS.isInNode()) {  // this means we are using node.js  
            for (i = 0; i < libraries.length; i++) {
                // store namespaces inside factories
                require("./" + libraries[i] + ".js");
            }
        } else {
            // in browser, we have an asynchronous load 
            // all the work is in fact performed in function arcs_module
            require(libraries);
        }
    };
    
    
    this.loadLibrary = function (libName, cbFunction) {
        var libUrl = libName, libActualName = libName;
        
        ARCS.Application.currentApplication = self;
        if (typeof libName !== "string") {
            libActualName = libName.name;
            libUrl = libName.url;
        }
                
        libraries.push(libActualName);
        if (ARCS.isInNode()) 
        { 
            require("./" + libUrl + ".js"); 
            if (cbFunction !== undefined) {
                cbFunction();
            }
        }
        else 
        { 
            require([libUrl], function () {
                if (cbFunction !== undefined) {
                    cbFunction();
                }
            }); 
        }
    };
    
    
    this.export = function() {
        var i;
        var description = { 
            context: {
                libraries: [],
                components: context
            }, 
            controller: context.getComponentName(controller), 
            sheets: sheets            
        } ;
        
        // first problem: when loaded by the editor, libraries are not the good ones
        description.context.libraries = libraries;        
        return description;
    };
    
    /*this.getContext = function () {
        return context;
    };*/
    
    this.getFactoryList = function() {
        return Object.keys(factories);
    };
    
    this.getSheetList = function() {
        return Object.keys(sheets);
    };
    
    this.getFactory = function (fName) {
        return factories[fName];
    };
    
    this.getSheet = function (sName) {
        return sheets[sName];
    };
    
    this.getComponentList = function () {
        return context.getComponentList();
    };
    
    this.getComponent = function (cName) {
        return context.getComponent(cName);
    };
    
    this.getComponentName = function (cmp) {
        return context.getComponentName(cmp);
    };
    
    
    this.addSheet = function (sName, sheet) {
        sheets[sName] = sheet;
        sheet.setContext(context);
    };

    this.setComponentValue = function (cName, cValue) {
        context[cName].value = cValue;        
    };
        
    this.addComponent = function (cName, cType, cValue) {
        var component;
        context[cName] = {};
        context[cName].type = cType;
        context[cName].value = cValue;
        
        var factory = factories[cType];
        if (factory !== undefined) {
            component = new factory(cValue);
        }
        context[cName].instance = component;        
    };
    
    this.removeSheet = function (sName) {
        delete sheets[sName];
    };
    
    this.removeComponent = function (cName) {
        delete context[cName];
    };
    
    checkFactories = function () {
        var i, cmpList = Object.keys(context);
        for (i = 0; i < cmpList.length; i++) {
            if (context.hasOwnProperty(cmpList[i])) {
                if (factories[context[cmpList[i]].type] === undefined) {
                    return;
                }
            }
        }
        autoStart = false;
        console.log("[ARCS] All factories are operational !");
        preProcess();
    };


    preProcess = function () {
        console.log("[ARCS] Starting application");
        // first, we should instanciate components
        var i, factory, instance, temp, sheetList, cmpList = Object.keys(context);
        for (i = 0; i < cmpList.length; i++) {
            factory = factories[context[cmpList[i]].type];
            //console.log(context[cmpList[i]].type);
            instance = new factory(context[cmpList[i]].value);
            context[cmpList[i]].instance = instance;
        }

        temp = context[controller].instance;
        controller = temp;
        // then we should work on sheets
        sheetList = Object.keys(sheets);
        for (i = 0; i < sheetList.length; i++) {
            temp = new ARCS.Sheet(context);
            temp.import(sheets[sheetList[i]], context);
            sheets[sheetList[i]] = temp;
        }

        ARCS.Component.connect(controller, "requestSheet", self, "setSheet");
        ARCS.Component.connect(controller, "requestTermination", self, "finish");
        controller.start();
    };
    
    
    this.setController = function (ctrlName) {
        controller = context[ctrlName].instance;
    };
    
    /**
     * Sets the current sheet of the application. This method is in fact designed
     * as a slot and may be triggered by a statemachine. If a sheet is already the
     * current one, then it is deactivated before activating this new sheet.
     * @param sheetName {string} name of the sheet to set as a current sheet.
     */
    this.setSheet = function (sheetname) {
        if (currentSheet) {
            sheets[currentSheet].deactivate();
        }
        currentSheet = sheetname;
        sheets[currentSheet].activate();
    };
    /**
     * This is the end my friend. This triggers the end of the application
     */
    this.finish = function () {
        if (currentSheet) {
            sheets[currentSheet].deactivate();
        }
    };
    /**
     * Imports a structured object describing the application. The structured object
     * may be described itself in a JSON format.
     * @param object {object} structured object describing an application.
     * 
     *   
     * @example
     * // JSON format of an application description
     * {
     *      context : {
     *              libraries : [ "library1", "library2"],
     *              components : [
     *                      // this could be also properties from context
     *                      name1: { type: "type", value: "value if needed" }
     *              ],
     *              constants : [
     *                      // the same observation applies here
     *                      name1: { representation : {JSON/objectRepresentation ? } }
     *              ]
     *      },
     *      controller : controllerId,
     *      sheets : {
     *              sheetId : {     
     *                      preconnections : [
     *                              { 
     *                                      destination: "id", 
     *                                      slot : "slot name",
     *                                      value : JSON/objectRepresentation ?
     *                              }, {...}, {...}
     *                      ],
     *                      postconnections : [
     *                              { 
     *                                      destination: "id", 
     *                                      slot : "slot name",
     *                                      value : JSON/objectRepresentation ?
     *                              }, {...}, {...}
     *                      ],
     *                      connections : [
     *                              {
     *                                      source: "id",
     *                                      destination: "id",
     *                                      slot: "slot name",
     *                                      signal: "signal name"
     *                              }, {...}, {...}
     *                      ],
     *                      cleanups : [
     *                              { 
     *                                      destination: "id", 
     *                                      slot : "slot name",
     *                                      value : JSON/objectRepresentation ?
     *                              }, {...}, {...}
     *                      ]
     *              },
     *              { ... }
     *      }
     * }
     * 
     */
    this.import = function (object) {
        libraries = object.context.libraries;
        context = new ARCS.Context(object.context.components);
        sheets = object.sheets;
        controller = object.controller;
        if (controller === undefined) {
            console.error("[ARCS] Undefined controller. Cannot start application.");
        }
    };

    /**
     * Registers a factory using a key. If a factory was previously existing using 
     * the same key, then it is overridden.
     * @param key {string} name of the factory
     * @param factory {object} component factory to register.
     */
    this.setFactory = function (key, factory) {
        //console.log("[ARCS] registered factory %s", key);
        factories[key] = factory;
        if (autoStart) { checkFactories(); }
    };

    this.setDependency = function (key) {
        dependencies[key] = {};
    };

    /**
     * Starts the application
     */
    this.start = function () {
        autoStart = true;
        loadLibraries();
    };
};

/**
 * Helper function that registers a factory from the global namespace to a given
 * application. This is defined in order to be callable by modules.
 */
ARCS.Application.setFactory = function (app, key, factory) {
    if (ARCS.Component.check(factory)) {
        app.setFactory(key, factory);
    } else {
        console.warn("[ARCS]",key,"is not a factory for an ARCS component.");
    }
};

ARCS.Application.setDependency = function (app, key) {
    app.setDependency(key);
};



ARCS.Component.create(ARCS.Application);
ARCS.Application.slot("setSheet");
ARCS.Application.slot("finish");

/**
 * definition of the main module function: 
 * it takes an anonymous function as a parameter
 * the anonymous function has one parameter: the object encompassing 
 * ARCS definitions (in order to able to use ARCS.Component.create, ...)
 * @param moduleDefinition {function} main function of the module. 
 * It should return a list of components
 * @param deps {mixed[]} dependencies
 */
arcs_module = function (moduleDefinition, deps) {
    var dependencies = [], loadedDependencies = 0, storeComponents, i;

    if (typeof module !== 'undefined') {
        if (module.parent.exports) {
            ARCS = module.exports; //.parent.exports;
        //console.log(module);
        }
    }

    dependencies[0] = ARCS;

    //console.log(ARCS);
    // one of the first problems that may arise is that we should solve dependencies
    // here before launching the application since module definition will depend on it !
    // it will depend wether we are using node.js or require.js !

    if (deps === undefined) { deps = []; }

    storeComponents = function (deps) {
        // coté asynchrone non défini
        //console.log("storing components");
        var mdef, p, actualDep, shimConfig;
        mdef = (typeof moduleDefinition === 'function') ?
                moduleDefinition.apply(this, deps) : moduleDefinition;

        if (mdef === undefined) {
            throw new Error("[ARCS] Your module is undefined. Did you forget to export components?\nCode of module follows:\n"+moduleDefinition);
        }

        //console.log("[ARCS] Loading factories:",mdef);
        for (p in mdef) {
             ARCS.Application.setFactory(ARCS.Application.currentApplication, p, mdef[p]);
        }

        // hack for require.js
        // ARCS is then defined as a require.js module.
        
        // It seems this hack is not needed anymore ! --> Great !
        /*if (ARCS.isInRequire()) {
            define(mdef);
        }*/
    };

    // for each dependency, do:       
    for (i = 0; i < deps.length; i++) {
        if (ARCS.isInNode()) {
            // here this is a synchronous load of the script
            dependencies[i] = require(deps[i]);
        } else {
            // here we should use require
            actualDep = deps[i];
            if (actualDep.name !== undefined) {
                shimConfig = { shim: {} };
                shimConfig.shim[actualDep.name] = { exports: actualDep.exports };
                if (actualDep.deps !== undefined) {
                    shimConfig.shim[actualDep.name].deps = actualDep.deps;
                }
                require.config(shimConfig);
                actualDep = actualDep.name;
            }
            //console.log("[ARCS] Loading dependency ",actualDep);
            
            // since actualDep and i may be moving, we will create a function
            // that creates a function (one more step and it will be currying
            var handleDep = function(depName, index) {
                    return function(dep) {
                        //console.log(depName,"loaded",index);
                        loadedDependencies++;
                        dependencies[index+1] = dep;
                        if (loadedDependencies >= deps.length) {
                            storeComponents(dependencies);
                        }
                    };
            };
            
            
            require([actualDep], 
                handleDep(actualDep,i),
                function (err) {
                    console.error(JSON.stringify(err));
                }
                );
        }
    }

    if (ARCS.isInNode() || deps.length === 0) { storeComponents(dependencies); }
};


// ARCS is then defined as a node.js module
if (!ARCS.isInNode()) {
    var exports = {};
}

exports.Component = ARCS.Component;
exports.Connection = ARCS.Connection;
exports.Invocation = ARCS.Invocation;
exports.Statemachine = ARCS.Statemachine;
exports.Sheet = ARCS.Sheet;
exports.Application = ARCS.Application;
exports.isInNode = ARCS.isInNode;
exports.isInRequire = ARCS.isInRequire;

// hack for require.js
// ARCS is then defined as a require.js module.
if (ARCS.isInRequire()) {
    //console.log("module ARCS in require.js");
    define(exports);
}
