arcs_module(function(ARCS) {
    /**
     * @class GeoLocator
     * @classdesc Uses the location provider of the browser to 
     * determine the geolocation of the device.
     * This component relies on the 
     * {@link http://www.w3.org/TR/geolocation-API/|Geolocation API}.
     */
    var GeoLocator = ARCS.Component.create(function() {
        var self = this;
        var geolocator = navigator.geolocation ;
        var watcher;
        var timeout = 1000;
        var enableHighAccuracy = false;

        var handlePosition = function(position) {
            self.emit("sendPosition",position.coords.longitude,position.coords.latitude);
            self.emit("sendCoordinates", position);
        };
        
        var handleError = function(positionError) {
            switch(positionError.code) {
                case positionError.TIMEOUT:
                    console.error("Timeout while geolocalizing", positionError.message);
                    break;
                    
                case positionError.PERMISSION_DENIED:
                    console.error("You must authorize geolocation", positionError.message);
                    break;
                    
                case positionError.POSITION_UNAVAILABLE:
                    console.error("Position is unavailable", positionError.message);
                    break;
            }
            self.emit("geolocationDisabled");
        };
        
        
        /**
         * Set the timeout (in milliseconds) between the call of 
         * {@link GeoLocator#getPosition|getPosition} or
         * {@link GeoLocator#watchPosition|watchPosition} and 
         * the sending of signal {@link GeoLocator#sendPosition|sendPosition}.
         * @param t {number} timeout duration in milliseconds
         * @function GeoLocator#setTimeout
         * @slot
         */
        this.setTimeout = function(t) {
            timeout = t;
        };
        
        this.enableHighAccuracy = function() {
            enableHighAccuracy = true;
        };
        
        this.disableHighAccuracy = function() {
            enableHighAccuracy = false;
        };
           
        /**
         * Retrieves the geolocation of the device. Use this function for a 
         * one shot retrieval. If it succeeds, signals {@link GeoLocator#sendPosition|sendPosition}
         * and {@link GeoLocator#sendCoordinates|sendCoordinates} are triggered.
         * Otherwise, signal {@link GeoLocator#geolocationDisabled|geolocationDisabled} is triggered.
         * @emits sendPosition
         * @emits sendCoordinates
         * @emits geolocationDisabled
         * @function GeoLocator#getPosition
         * @slot
         */
        this.getPosition = function() {
            if (geolocator) {
                geolocator.getCurrentPosition(handlePosition,handleError, 
                    { timeout: timeout, enableHighAccuracy : enableHighAccuracy} );                
            } else {
                this.emit("geolocationDisabled");
            }
        };
        
        
        /**
         * Retrieves the geolocation of the device. Use this function if you
         * want to continuously obtain the geolocation of the device.
         * If it succeeds, signals {@link GeoLocator#sendPosition|sendPosition}
         * and {@link GeoLocator#sendCoordinates|sendCoordinates} are triggered.
         * Otherwise, signal {@link GeoLocator#geolocationDisabled|geoLocationDisabled} is triggered.
         * You can use {@link clearWatch} to stop watching the position.
         * @emits sendPosition
         * @emits sendCoordinates
         * @emits geolocationDisabled
         * @function GeoLocator#watchPosition
         * @slot
         */
        this.watchPosition= function() {
            if (geolocator) {
                if (watcher) {
                    geolocator.clearWatch(watcher);                    
                }
                watcher = geolocator.watchPosition(handlePosition, handleError, 
                     { timeout: timeout, enableHighAccuracy : enableHighAccuracy} );                 
            } else {
                this.emit("geolocationDisabled");
            }
        };
        
        
        /**
         * Ends geolocation queries.
         * @see {@link GeoLocator#watchPosition|watchPosition} 
         * @function GeoLocator#clearWatch
         * @slot 
         */        
        this.clearWatch = function() {
            if (geolocator) {
                if (watcher) {
                    geolocator.clearWatch(watcher);
                }
            }
        };
        
        /**
         * Sends the position of the device (longitude and latitude)
         * @param lon {number} longitude of the geolocation 
         * @param lat {number} latitude of the geolocation
         * @function GeoLocator#sendPosition
         * @signal
         */
        
        /**
         * Sends the coordinates of the device using the coordinates object
         * proposed in the specification of the geolocation API.
         * @param coords {object} coordinates of the device. 
         * @function GeoLocator#sendCoordinates
         * @signal
         */

        /**
         * Notifies that the geolocation service is disabled 
         * @function GeoLocator#geolocationDisabled
         * @signal
         */
        
    },
    [
        "getPosition","watchPosition","clearWatch","setTimeout",
        "enableHighAccuracy", "disableHighAccuracy"
        
    ], // liste slots
    ["sendPosition","sendCoordinates","geolocationDisabled"] // liste de signaux    
    );
    
    



    
    return { GeoLocator : GeoLocator } ; 
});